<?php

declare(strict_types=1);

namespace Drupal\auditfiles\Form;

use Drupal\auditfiles\Auditor\AuditFilesNotOnServer;
use Drupal\auditfiles\Services\AuditFilesConfigInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\ConfirmFormHelper;
use Drupal\Core\Form\ConfirmFormInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Not on server.
 *
 * Lists files in the database, but not in the "files" directory.
 *
 * @internal
 *   There is no extensibility promise for this class. Use form alter hooks to
 *   make customisations.
 */
final class AuditFilesNotOnServerForm extends FormBase implements ConfirmFormInterface {

  /**
   * Constructs a new AuditFilesNotOnServerForm.
   */
  final public function __construct(
    protected AuditFilesConfigInterface $auditFilesConfig,
    protected AuditFilesNotOnServer $auditFilesNotOnServer,
    protected PagerManagerInterface $pagerManager,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FileSystemInterface $fileSystem,
    protected DateFormatterInterface $dateFormatter,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('auditfiles.config'),
      $container->get('auditfiles.not_on_server'),
      $container->get('pager.manager'),
      $container->get('entity_type.manager'),
      $container->get('file_system'),
      $container->get('date.formatter'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'audit_files_not_on_server';
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): TranslatableMarkup {
    return $this->t('This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Confirm');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelText(): TranslatableMarkup {
    return $this->t('Cancel');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormName(): string {
    return 'audit_files_not_on_server';
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('auditfiles.audit_files_notonserver');
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t("Do you want to delete following record");
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $storage = &$form_state->getStorage();
    if (isset($storage['confirm'])) {
      $values = $form_state->getValue('files');
      $form['changelist'] = [
        '#prefix' => '<ul>',
        '#suffix' => '</ul>',
        '#tree' => TRUE,
      ];
      // Prepare the list of items to present to the user.
      if (!empty($values)) {
        foreach ($values as $file_id) {
          if (!empty($file_id)) {
            $file = $this->entityTypeManager->getStorage('file')->load($file_id);
            if (!empty($file)) {
              $form['changelist'][$file_id] = [
                '#type' => 'hidden',
                '#value' => $file_id,
                '#prefix' => '<li><strong>' . $file->getFilename() . '</strong> ' . $this->t('and all usages will be deleted from the database.'),
                '#suffix' => "</li>\n",
              ];
            }
          }
          else {
            unset($form_state->getValue('files')[$file_id]);
          }
        }
      }
      $form['#title'] = $this->t('Delete these items from the database?');
      $form['#attributes']['class'][] = 'confirmation';

      $form['actions'] = [
        '#type' => 'actions',
      ];

      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->getConfirmText(),
        '#button_type' => 'primary',
      ];

      $form['actions']['cancel'] = ConfirmFormHelper::buildCancelLink($this, $this->getRequest());

      if (!isset($form['#theme'])) {
        $form['#theme'] = 'confirm_form';
      }
      return $form;
    }

    foreach ($this->auditFilesNotOnServer->getReferences() as $reference) {
      $file = $reference->getFile() ?? throw new \LogicException('The file_managed row exists so this should be loadable.');
      $dateFormat = $this->auditFilesConfig->getReportOptionsDateFormat();
      $rows[$reference->getId()] = [
        'fid' => $file->id(),
        'uid' => $file->getOwnerId(),
        'filename' => $file->getFilename(),
        'uri' => $file->getFileUri(),
        'path' => $this->fileSystem->realpath($file->getFileUri()),
        'filemime' => $file->getMimeType(),
        'filesize' => $file->getSize() ? \number_format((float) $file->getSize()) : NULL,
        'datetime' => $this->dateFormatter->format($file->getCreatedTime(), $dateFormat),
        'status' => $file->isPermanent() ? $this->t('Permanent') : $this->t('Temporary'),
      ];
    }

    // Set up the pager.
    if (!empty($rows)) {
      $items_per_page = $this->auditFilesConfig->getReportOptionsItemsPerPage();
      if (!empty($items_per_page)) {
        $current_page = $this->pagerManager->createPager(count($rows), $items_per_page)->getCurrentPage();
        // Break the total data set into page sized chunks.
        $pages = array_chunk($rows, $items_per_page, TRUE);
      }
    }
    // Define the form Setup the record count and related messages.
    $maximum_records = $this->auditFilesConfig->getReportOptionsMaximumRecords();
    if (!empty($rows)) {
      if ($maximum_records > 0) {
        $file_count_message = $this->t('Found at least @count file entities with missing files on disk.');
      }
      else {
        $file_count_message = $this->t('Found @count file entities with missing files on disk.');
      }
      $form_count = $this->formatPlural(count($rows), $this->t('Found 1 file entity with missing file on disk.'), $file_count_message);
    }
    else {
      $form_count = $this->t('Found no file entities with missing files on disk.');
    }

    // Create the form table.
    $form['files'] = [
      '#type' => 'tableselect',
      '#header' => [
        'fid' => [
          'data' => t('File ID'),
        ],
        'uid' => [
          'data' => t('User ID'),
        ],
        'filename' => [
          'data' => t('Name'),
        ],
        'uri' => [
          'data' => t('URI'),
        ],
        'path' => [
          'data' => t('Path'),
        ],
        'filemime' => [
          'data' => t('MIME'),
        ],
        'filesize' => [
          'data' => t('Size'),
        ],
        'datetime' => [
          'data' => t('When added'),
        ],
        'status' => [
          'data' => t('Status'),
        ],
      ],
      '#empty' => $this->t('No items found.'),
      '#prefix' => '<div><em>' . $form_count . '</em></div>',
    ];
    // Add the data.
    if (!empty($rows) && !empty($pages)) {
      $form['files']['#options'] = $pages[$current_page];
    }
    elseif (!empty($rows)) {
      $form['files']['#options'] = $rows;
    }
    else {
      $form['files']['#options'] = [];
    }
    // Add any action buttons.
    if (!empty($rows)) {
      $form['actions'] = ['#type' => 'actions'];
      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Delete selected items from the database'),
        '#submit' => ['::submissionHandlerDeleteFromDb'],
      ];
      $form['pager'] = ['#type' => 'pager'];
    }
    return $form;
  }

  /**
   * Delete record to database.
   */
  public function submissionHandlerDeleteFromDb(array &$form, FormStateInterface $form_state): void {
    if (!empty($form_state->getValue('files'))) {
      foreach ($form_state->getValue('files') as $file_id) {
        if (!empty($file_id)) {
          $storage = [
            'files' => $form_state->getValue('files'),
            'op' => 'del',
            'confirm' => TRUE,
          ];
          $form_state->setStorage($storage);
          $form_state->setRebuild();
        }
      }
      if (!isset($storage)) {
        $this->messenger()->addError(
          $this->t('No items were selected to operate on.')
        );
      }
    }
  }

  /**
   * Delete record from database confirm.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    \batch_set($this->auditFilesNotOnServer->auditfilesNotOnServerBatchDeleteCreateBatch($form_state->getValue('changelist'))->toArray());
  }

}
