<?php

declare(strict_types=1);

namespace Drupal\auditfiles\Form;

use Drupal\auditfiles\Auditor\AuditFilesReferencedNotUsed;
use Drupal\auditfiles\Reference\FileFieldReference;
use Drupal\auditfiles\Services\AuditFilesConfigInterface;
use Drupal\Core\Form\ConfirmFormHelper;
use Drupal\Core\Form\ConfirmFormInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Referenced not used.
 *
 * List files referenced in content, but not in the file_usage table.
 *
 * @internal
 *   There is no extensibility promise for this class. Use form alter hooks to
 *   make customisations.
 */
final class AuditFilesReferencedNotUsedForm extends FormBase implements ConfirmFormInterface {

  /**
   * Constructs a new AuditFilesReferencedNotUsedForm.
   */
  final public function __construct(
    protected AuditFilesConfigInterface $auditFilesConfig,
    protected AuditFilesReferencedNotUsed $filesReferencedNotUsed,
    protected PagerManagerInterface $pagerManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('auditfiles.config'),
      $container->get('auditfiles.referenced_not_used'),
      $container->get('pager.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'audit_files_referenced_not_used';
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): TranslatableMarkup {
    return $this->t('This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Confirm');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelText(): TranslatableMarkup {
    return $this->t('Cancel');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormName(): string {
    return 'audit_files_referenced_not_used';
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('auditfiles.audit_files_referencednotused');
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t("Do you want to delete following record");
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $storage = &$form_state->getStorage();
    if (isset($storage['confirm'])) {
      $values = $form_state->getValue('files');
      $form['changelist'] = [
        '#prefix' => '<ul>',
        '#suffix' => '</ul>',
        '#tree' => TRUE,
      ];
      if (!empty($values)) {
        foreach ($values as $reference_id) {
          if (!empty($reference_id)) {
            ['fileId' => $fileId] = FileFieldReference::reverseRowKey($reference_id);
            if ($storage['op'] == 'add') {
              $message = $this->t('will be added to the file usage table.');
            }
            elseif ($storage['op'] == 'delete') {
              $message = $this->t('will be deleted from the content.');
            }
            $form['changelist'][$reference_id] = [
              '#type' => 'hidden',
              '#value' => $reference_id,
              '#prefix' => '<li>' . $this->t('File ID') . ' <strong>' . $fileId . '</strong> ' . $message,
              '#suffix' => "</li>\n",
            ];
          }
          else {
            unset($form_state->getValue('files')[$reference_id]);
          }
        }
      }
      if ($storage['op'] == 'add') {
        $form['#title'] = $this->t('Add these files to the database?');
      }
      elseif ($storage['op'] == 'delete') {
        $form['#title'] = $this->t('Delete these files from the server?');
      }
      $form['actions'] = [
        '#type' => 'actions',
      ];
      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->getConfirmText(),
        '#button_type' => 'primary',
      ];
      $form['actions']['cancel'] = ConfirmFormHelper::buildCancelLink($this, $this->getRequest());
      if (!isset($form['#theme'])) {
        $form['#theme'] = 'confirm_form';
      }
      return $form;
    }

    foreach ($this->filesReferencedNotUsed->getReferences() as $reference) {
      $rows[$reference->rowKey()] = $this->fileRow($reference);
    }

    if (!empty($rows)) {
      $items_per_page = $this->auditFilesConfig->getReportOptionsItemsPerPage();
      if (!empty($items_per_page)) {
        $current_page = $this->pagerManager->createPager(count($rows), $items_per_page)->getCurrentPage();
        $pages = array_chunk($rows, $items_per_page, TRUE);
      }
    }
    $maximum_records = $this->auditFilesConfig->getReportOptionsMaximumRecords();
    if (!empty($rows)) {
      if ($maximum_records > 0) {
        $file_count_message = $this->t('Found at least @count files referenced in content not in the file_usage table.');
      }
      else {
        $file_count_message = $this->t('Found @count files referenced in content not in the file_usage table.');
      }
      $form_count = $this->formatPlural(count($rows), $this->t('Found 1 file referenced in content that is not in the file_usage table.'), $file_count_message);
    }
    else {
      $form_count = $this->t('Found no files referenced in content not in the file_usage table.');
    }
    $form['files'] = [
      '#type' => 'tableselect',
      '#header' => [
        'file_id' => [
          'data' => t('File ID'),
        ],
        'entity_type' => [
          'data' => t('Referencing entity type'),
        ],
        'entity_id_display' => [
          'data' => t('Referencing entity ID'),
        ],
        'field' => [
          'data' => t('Field referenced in'),
        ],
        'uri' => [
          'data' => t('URI'),
        ],
        'filemime' => [
          'data' => t('MIME'),
        ],
        'filesize' => [
          'data' => t('Size (in bytes)'),
        ],
      ],
      '#empty' => $this->t('No items found.'),
      '#prefix' => '<div><em>' . $form_count . '</em></div>',
    ];
    // Add the data.
    if (!empty($rows) && !empty($pages)) {
      $form['files']['#options'] = $pages[$current_page];
    }
    elseif (!empty($rows)) {
      $form['files']['#options'] = $rows;
    }
    else {
      $form['files']['#options'] = [];
    }
    // Add any action buttons.
    if (!empty($rows)) {
      $form['actions'] = ['#type' => 'actions'];
      $form['actions']['add'] = [
        '#type' => 'submit',
        '#value' => $this->t('Add selected items to the file_usage table'),
        '#submit' => ['::submissionHandlerAddToFile'],
      ];
      $text = $this->t('or');
      $form['actions']['markup'] = [
        '#markup' => '&nbsp;' . $text . '&nbsp;',
      ];
      $form['actions']['delete'] = [
        '#type' => 'submit',
        '#value' => $this->t('Delete selected references'),
        '#submit' => ['::submissionHandlerDeleteFromFileUsage'],
      ];
      $form['pager'] = ['#type' => 'pager'];
    }
    return $form;
  }

  /**
   * Submit form.
   */
  public function submissionHandlerAddToFile(array &$form, FormStateInterface $form_state): void {
    if (!empty($form_state->getValue('files'))) {
      foreach ($form_state->getValue('files') as $file_id) {
        if (!empty($file_id)) {
          $storage = [
            'files' => $form_state->getValue('files'),
            'confirm' => TRUE,
            'op' => 'add',
          ];
          $form_state->setStorage($storage);
          $form_state->setRebuild();
        }
      }
      if (!isset($storage)) {
        $this->messenger()->addError(
          $this->t('No items were selected to operate on.')
        );
      }
    }
  }

  /**
   * Submit form.
   */
  public function submissionHandlerDeleteFromFileUsage(array &$form, FormStateInterface $form_state): void {
    if (!empty($form_state->getValue('files'))) {
      foreach ($form_state->getValue('files') as $file_id) {
        if (!empty($file_id)) {
          $storage = [
            'files' => $form_state->getValue('files'),
            'confirm' => TRUE,
            'op' => 'delete',
          ];
          $form_state->setStorage($storage);
          $form_state->setRebuild();
        }
      }
      if (!isset($storage)) {
        $this->messenger()->addError(
          $this->t('No items were selected to operate on.')
        );
      }
    }
  }

  /**
   * Delete record from files.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $storage = &$form_state->getStorage();
    if ($storage['op'] == 'add') {
      \batch_set($this->filesReferencedNotUsed->auditfilesReferencedNotUsedBatchAddCreateBatch($form_state->getValue('changelist'))->toArray());
    }
    else {
      \batch_set($this->filesReferencedNotUsed->auditfilesReferencedNotUsedBatchDeleteCreateBatch($form_state->getValue('changelist'))->toArray());
    }
  }

  /**
   * Retrieves information about an individual file from the database.
   *
   * @return array
   *   The row for the table on the report, with the file's
   *   information formatted for display.
   */
  protected function fileRow(FileFieldReference $reference): array {
    $row = [
      'file_id' => $reference->getTargetFileId(),
      'entity_type' => $reference->getSourceEntityTypeId(),
      'bundle' => ['data' => $reference->bundle, 'hidden' => TRUE],
      'entity_id' => ['data' => $reference->getSourceEntityId(), 'hidden' => TRUE],
    ];

    $sourceFile = \Drupal::entityTypeManager()
      ->getStorage($reference->getSourceEntityTypeId())
      ->load($reference->getSourceEntityId());
    $row['entity_id_display'] =
      ($sourceFile !== NULL && $sourceFile->hasLinkTemplate('canonical'))
        ? $sourceFile->toLink(rel: 'canonical')
        : $reference->getSourceEntityId();

    $row['field'] = $reference->table . '.' . $reference->column;
    $row['table'] = ['data' => $reference->table, 'hidden' => TRUE];

    // If there is a file in the file_managed table, add some of that
    // information to the row, too.
    $file_managed = File::load($reference->getTargetFileId());
    if ($file_managed !== NULL) {
      $this->fileUrlGenerator = \Drupal::service('file_url_generator');
      $row['uri'] = Link::fromTextAndUrl($file_managed->getFileuri(), Url::fromUri($this->fileUrlGenerator->generateAbsoluteString($file_managed->getFileuri()), ['attributes' => ['target' => '_blank']]));
      $row['filename'] = [
        'data' => $file_managed->getFilename(),
        'hidden' => TRUE,
      ];
      $row['filemime'] = $file_managed->getMimeType();
      $row['filesize'] = $file_managed->getSize();
    }
    else {
      $row['uri'] = $this->t('Target file was deleted');
      $row['filename'] = ['data' => '', 'hidden' => TRUE];
      $row['filemime'] = '--';
      $row['filesize'] = '--';
    }

    return $row;
  }

}
