<?php

declare(strict_types=1);

namespace Drupal\auditfiles\Form;

use Drupal\auditfiles\Auditor\AuditFilesUsedNotManaged;
use Drupal\auditfiles\Services\AuditFilesConfigInterface;
use Drupal\Core\Form\ConfirmFormHelper;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Usage with missing files.
 *
 * Lists entries in the file usage table, where the file entity does not exist.
 *
 * @internal
 *   There is no extensibility promise for this class. Use form alter hooks to
 *   make customisations.
 */
final class AuditFilesUsedNotManagedForm extends FormBase implements AuditFilesAuditorFormInterface {

  use AuditFilesAuditorFormTrait;

  /**
   * Constructs a new AuditFilesUsedNotManagedForm.
   */
  final public function __construct(
    protected AuditFilesConfigInterface $auditFilesConfig,
    protected AuditFilesUsedNotManaged $filesUsedNotManaged,
    protected PagerManagerInterface $pagerManager
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('auditfiles.config'),
      $container->get('auditfiles.used_not_managed'),
      $container->get('pager.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'audit_files_used_not_managed';
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): TranslatableMarkup {
    return $this->t('This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Confirm');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelText(): TranslatableMarkup {
    return $this->t('Cancel');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormName(): string {
    return 'audit_files_used_not_managed';
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('auditfiles.audit_files_usednotmanaged');
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t("Do you want to delete following record");
  }

  /**
   * {@inheritdoc}
   */
  public function buildListForm(array $form, FormStateInterface $form_state): array {
    foreach ($this->filesUsedNotManaged->getReferences() as $reference) {
      // Get the file information for the specified file ID from the database.
      $url = Url::fromUri('internal:/' . $reference->getEntityTypeId() . '/' . $reference->getEntityId());
      $result = Link::fromTextAndUrl($reference->getEntityTypeId() . '/' . $reference->getEntityId(), $url)->toString();
      $rows[$reference->getFileId()] = [
        'fid' => $reference->getFileId(),
        'module' => $this->t('@module module', ['@module' => $reference->getModule()]),
        'id' => $result,
        'count' => $reference->getCount(),
      ];
    }

    // Set up the pager.
    if (count($rows) > 0) {
      $items_per_page = $this->auditFilesConfig->getReportOptionsItemsPerPage();
      if (!empty($items_per_page)) {
        $current_page = $this->pagerManager->createPager(count($rows), $items_per_page)->getCurrentPage();
        $pages = array_chunk($rows, $items_per_page, TRUE);
      }
    }

    // Setup the record count and related messages.
    $maximum_records = $this->auditFilesConfig->getReportOptionsMaximumRecords();
    if (!empty($rows)) {
      if ($maximum_records > 0) {
        $file_count_message = $this->t('Found at least @count entries in the file usage table where file is missing.');
      }
      else {
        $file_count_message = $this->t('Found @count entries in the file usage table where file is missing.');
      }
      $form_count = $this->formatPlural(count($rows), $this->t('Found 1 entries in the file usage table where file is missing.'), $file_count_message);
    }
    else {
      $form_count = $this->t('Found no entries in the file usage table where file is missing.');
    }
    // Create the form table.
    $form['files'] = [
      '#type' => 'tableselect',
      '#header' => [
        'fid' => [
          'data' => $this->t('Missing-File ID'),
        ],
        'module' => [
          'data' => $this->t('Used by'),
        ],
        'id' => [
          'data' => $this->t('Used in'),
        ],
        'count' => [
          'data' => $this->t('Count'),
        ],
      ],
      '#empty' => $this->t('No items found.'),
      '#prefix' => '<div><em>' . $form_count . '</em></div>',
    ];

    // Add the data.
    if (!empty($rows) && !empty($pages)) {
      $form['files']['#options'] = $pages[$current_page];
    }
    elseif (!empty($rows)) {
      $form['files']['#options'] = $rows;
    }
    else {
      $form['files']['#options'] = [];
    }

    // Add any action buttons.
    if (!empty($rows)) {
      $form['actions'] = ['#type' => 'actions'];
      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Delete selected items from the file_usage table'),
      ];
      $form['pager'] = ['#type' => 'pager'];
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfirmForm(array $form, FormStateInterface $form_state): array {
    $values = $form_state->getValue('files');
    $form['changelist'] = [
      '#prefix' => '<ul>',
      '#suffix' => '</ul>',
      '#tree' => TRUE,
    ];
    // Prepare the list of items to present to the user.
    if (!empty($values)) {
      foreach ($values as $file_id) {
        if (!empty($file_id)) {
          $form['changelist'][$file_id] = [
            '#type' => 'hidden',
            '#value' => $file_id,
            '#prefix' => '<li>' . $this->t('File ID') . ' <strong>' . $file_id . '</strong> ' . $this->t('will be deleted from the file_usage table.'),
            '#suffix' => "</li>\n",
          ];
        }
        else {
          unset($form_state->getValue('files')[$file_id]);
        }
      }
    }
    $form['#title'] = $this->t('Delete these items from the file_usage table?');
    $form['#attributes']['class'][] = 'confirmation';
    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->getConfirmText(),
      '#button_type' => 'primary',
      '#submit' => ['::confirmSubmissionHandlerDeleteFile'],
    ];
    $form['actions']['cancel'] = ConfirmFormHelper::buildCancelLink($this, $this->getRequest());
    if (!isset($form['#theme'])) {
      $form['#theme'] = 'confirm_form';
    }
    return $form;
  }

  /**
   * Submit handler for confirmation.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    if (!empty($form_state->getValue('files'))) {
      foreach ($form_state->getValue('files') as $file_id) {
        if (!empty($file_id)) {
          $storage = [
            'files' => $form_state->getValue('files'),
            'confirm' => TRUE,
          ];
          $form_state->setStorage($storage);
          $form_state->setRebuild();
        }
      }
      if (!isset($storage)) {
        $this->messenger()->addError(
          $this->t('No items were selected to operate on.')
        );
      }
    }
  }

  /**
   * Submit handler for confirmation.
   */
  public function confirmSubmissionHandlerDeleteFile(array &$form, FormStateInterface $form_state): void {
    \batch_set($this->filesUsedNotManaged->auditfilesUsedNotManagedBatchDeleteCreateBatch($form_state->getValue('changelist'))->toArray());
  }

}
