<?php

namespace Drupal\auditfiles\Form;

use Drupal\auditfiles\Auditor\AuditFilesUsedNotReferenced;
use Drupal\auditfiles\Services\AuditFilesConfigInterface;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfirmFormHelper;
use Drupal\Core\Form\ConfirmFormInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Pager\PagerManagerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Used not referenced.
 *
 * List files in the file_usage table, but not referenced in content.
 *
 * @internal
 *   There is no extensibility promise for this class. Use form alter hooks to
 *   make customisations.
 */
final class AuditFilesUsedNotReferencedForm extends FormBase implements ConfirmFormInterface {

  /**
   * Constructs a new AuditFilesUsedNotReferencedForm.
   */
  final public function __construct(
    protected AuditFilesConfigInterface $auditFilesConfig,
    protected AuditFilesUsedNotReferenced $filesUsedNotReferenced,
    protected PagerManagerInterface $pagerManager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('auditfiles.config'),
      $container->get('auditfiles.used_not_referenced'),
      $container->get('pager.manager'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'audit_files_used_not_referenced';
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): TranslatableMarkup {
    return $this->t('This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Confirm');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelText(): TranslatableMarkup {
    return $this->t('Cancel');
  }

  /**
   * {@inheritdoc}
   */
  public function getFormName(): string {
    return 'audit_files_used_not_referenced';
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('auditfiles.audit_files_usednotreferenced');
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t("Do you want to delete following record");
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $storage = &$form_state->getStorage();
    if (isset($storage['confirm'])) {
      $values = $form_state->getValue('files');
      $form['changelist'] = [
        '#prefix' => '<ul>',
        '#suffix' => '</ul>',
        '#tree' => TRUE,
      ];
      // Prepare the list of items to present to the user.
      if (!empty($values)) {
        foreach ($values as $file_id) {
          if (!empty($file_id)) {
            $file = $this->entityTypeManager->getStorage('file')->load($file_id);
            if (!empty($file)) {
              $form['changelist'][$file_id] = [
                '#type' => 'hidden',
                '#value' => $file_id,
                '#prefix' => '<li><strong>' . $file->getFilename() . '</strong> ' . $this->t('will be deleted from the file_usage table.'),
                '#suffix' => "</li>",
              ];
            }
          }
          else {
            unset($form_state->getValue('files')[$file_id]);
          }
        }
      }
      $form['#title'] = $this->t('Delete these items from the file_usage table?');
      $form['#attributes']['class'][] = 'confirmation';
      $form['actions'] = ['#type' => 'actions'];
      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->getConfirmText(),
        '#button_type' => 'primary',
        '#submit' => ['::confirmSubmissionHandlerDeleteFile'],
      ];
      $form['actions']['cancel'] = ConfirmFormHelper::buildCancelLink($this, $this->getRequest());
      if (!isset($form['#theme'])) {
        $form['#theme'] = 'confirm_form';
      }
      return $form;
    }

    $rows = array_reduce(\iterator_to_array($this->filesUsedNotReferenced->getReferences()), function (?array $rows, int $fileId) {
      $rows[$fileId] = $this->fileRow($fileId);
      return $rows;
    }, []);

    // Set up the pager.
    if (count($rows) > 0) {
      $items_per_page = $this->auditFilesConfig->getReportOptionsItemsPerPage();
      if (!empty($items_per_page)) {
        $current_page = $this->pagerManager->createPager(count($rows), $items_per_page)->getCurrentPage();
        // Break the total data set into page sized chunks.
        $pages = array_chunk($rows, $items_per_page, TRUE);
      }
    }
    // Setup the record count and related messages.
    $maximum_records = $this->auditFilesConfig->getReportOptionsMaximumRecords();
    if (count($rows) > 0) {
      if ($maximum_records > 0) {
        $file_count_message = $this->t('Found at least @count files in the file_usage table not referenced in content.');
      }
      else {
        $file_count_message = $this->t('Found @count files in the file_usage table not referenced in content.');
      }
      $form_count = $this->formatPlural(count($rows), $this->t('Found 1 file in the file_usage table that is not referenced in content.'), $file_count_message);
    }
    else {
      $form_count = $this->t('Found no files in the file_usage table not referenced in content.');
    }
    // Create the form table.
    $form['files'] = [
      '#type' => 'tableselect',
      '#header' => [
        'fid' => ['data' =>$this->t('File ID')],
        'uri' => ['data' =>$this->t('File URI')],
        'usage' => ['data' =>$this->t('Usages')],
      ],
      '#empty' => $this->t('No items found.'),
      '#prefix' => '<div><em>' . $form_count . '</em></div>',
    ];
    // Add the data.
    if (!empty($rows) && !empty($pages)) {
      $form['files']['#options'] = $pages[$current_page];
    }
    elseif (!empty($rows)) {
      $form['files']['#options'] = $rows;
    }
    else {
      $form['files']['#options'] = [];
    }
    // Add any action buttons.
    if (!empty($rows)) {
      $form['actions'] = ['#type' => 'actions'];
      $form['actions']['submit'] = [
        '#type' => 'submit',
        '#value' => $this->t('Delete selected items from the file_usage table'),
      ];
      $form['pager'] = ['#type' => 'pager'];
    }
    return $form;
  }

  /**
   * Submit for confirmation.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    if (!empty($form_state->getValue('files'))) {
      foreach ($form_state->getValue('files') as $file_id) {
        if (!empty($file_id)) {
          $storage = [
            'files' => $form_state->getValue('files'),
            'confirm' => TRUE,
          ];
          $form_state->setStorage($storage);
          $form_state->setRebuild();
        }
      }
      if (!isset($storage)) {
        $this->messenger()->addError($this->t('No items were selected to operate on.'));
      }
    }
  }

  /**
   * Submit form after confirmation.
   */
  public function confirmSubmissionHandlerDeleteFile(array &$form, FormStateInterface $form_state): void {
    \batch_set($this->filesUsedNotReferenced->auditfilesUsedNotReferencedBatchDeleteCreateBatch($form_state->getValue('changelist'))->toArray());
  }

  /**
   * Retrieves information about an individual file from the database.
   *
   * @param int $file_id
   *   The ID of the file to prepare for display.
   *
   * @return array
   *   The row for the table on the report, with the file's
   *   information formatted for display.
   */
  protected function fileRow(int $file_id): array {
    $file = File::load($file_id);
    if ($file === NULL) {
      $url = Url::fromUri('internal:/admin/reports/auditfiles/usednotmanaged');
      $result_link = Link::fromTextAndUrl(t('Used not managed'), $url)->toString();
      return [
        'fid' => [
          'colspan' => 3,
          'data' => $this->t('This file is not listed in the file_managed table. See the "%usednotmanaged" report.', ['%usednotmanaged' => $result_link]),
        ],
      ];
    }

    $usage = '<ul>';
    $results =  \Drupal::database()->query("SELECT * FROM {file_usage} WHERE fid = $file_id");
    foreach ($results as $file_usage) {
      $used_by = $file_usage->module;
      $type = $file_usage->type;
      $url = Url::fromUri('internal:/node/' . $file_usage->id);
      $result_link = Link::fromTextAndUrl($file_usage->id, $url)->toString();
      $used_in = ($file_usage->type == 'node') ? $result_link : $file_usage->id;
      $times_used = $file_usage->count;
      $usage .= '<li>' .$this->t(
          'Used by module: %used_by, as object type: %type, in content ID: %used_in; Times used: %times_used',
          [
            '%used_by' => $used_by,
            '%type' => $type,
            '%used_in' => $used_in,
            '%times_used' => $times_used,
          ]
        ) . '</li>';
    }
    $usage .= '</ul>';
    $usage = new FormattableMarkup($usage, []);

    /** @var \Drupal\Core\File\FileUrlGeneratorInterface $fileUrlGenerator */
    $fileUrlGenerator = \Drupal::service('file_url_generator');

    return [
      'fid' => $file_id,
      'uri' => Link::fromTextAndUrl($file->getFileUri(), Url::fromUri($fileUrlGenerator->generateAbsoluteString($file->getFileUri()), ['attributes' => ['target' => '_blank']])),
      'usage' => $usage,
    ];
  }

}
