<?php

declare(strict_types=1);

namespace Drupal\auditfiles\Reference;

/**
 * Represents an entry from a file field row.
 *
 * This includes derivative types of FileItem, such as ImageItem.
 *
 * @see \Drupal\file\Plugin\Field\FieldType\FileItem
 * @see \Drupal\image\Plugin\Field\FieldType\ImageItem
 */
final class FileFieldReference implements ReferenceInterface {

  /**
   * Constructs a new FileFieldReference.
   */
  private function __construct(
    public mixed $table,
    public mixed $column,
    public string|int $entityId,
    public int $fileId,
    public string $entityTypeId,
    public string $bundle,
  ) {
  }

  public static function create(
    mixed $table,
    mixed $column,
    string|int $entityId,
    int $fileId,
    string $entityTypeId,
    string $bundle,
  ): static {
    return new static($table, $column, $entityId, $fileId, $entityTypeId, $bundle);
  }

  /**
   * Get the row key.
   *
   * @return string
   *   The row key. Warning: the format of this may be used at any time so do
   *   not serialise/save this value.
   */
  public function rowKey(): string {
    return \sprintf('%s.%s.%d.%s.%s', $this->table, $this->column, $this->entityId, $this->entityTypeId, $this->fileId);
  }

  public function getSourceEntityId(): string|int {
    return $this->entityId;
  }

  public function getSourceEntityTypeId(): string {
    return $this->entityTypeId;
  }

  public function getSourceBundle(): string {
    return $this->bundle;
  }

  public function getTargetFileId(): int {
    return $this->fileId;
  }

  /**
   * @param string $key
   *   A key, as provided by rowKey().
   *
   * @return array{table: string, column: string, entityId: string, entityTypeId: string, fileId: string}
   */
  public static function reverseRowKey(string $key): array {
    return \array_combine(
      ['table', 'column', 'entityId', 'entityTypeId', 'fileId'],
      \explode('.', $key)
    );
  }

  public function __toString(): string {
    return sprintf('Reference at %s from entity %s #%s to file #%s', $this->table, $this->entityTypeId, $this->entityId, $this->fileId);
  }

}
