<?php

namespace Drupal\augmentor\Plugin\Action;

use Drupal\augmentor\AugmentorManager;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Action\ConfigurableActionBase;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Augmentor Minimal Action plugin.
 *
 * @Action(
 *   id = "entity:augmentor_action_minimal",
 *   action_label = @Translation("Augmentor Minimal"),
 *   deriver = "Drupal\augmentor\Plugin\Action\Derivative\AugmentorActionDeriver",
 * )
 */
class AugmentorActionMinimal extends ConfigurableActionBase implements ContainerFactoryPluginInterface {

  /**
   * The entity field manager.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $entityFieldManager;

  /**
   * The augmentor plugin manager.
   *
   * @var \Drupal\augmentor\AugmentorManager
   */
  protected $augmentorManager;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityFieldManagerInterface $entity_field_manager, AugmentorManager $augmentor_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityFieldManager = $entity_field_manager;
    $this->augmentorManager = $augmentor_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_field.manager'),
      $container->get('plugin.manager.augmentor.augmentors')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute($object = NULL) {
    $augmentor = $this->augmentorManager->getAugmentor($this->configuration['augmentor']);
    $input = '';
    for ($i = 1; $i <= 10; $i++) {
      $source_field_key = 'source_field_' . $i;
      if ($this->configuration[$source_field_key]) {
        $source_field = $this->configuration[$source_field_key];
        if ($object->hasField($source_field)) {
          $input .= $object->get($source_field)->getString();
        }
      }
    }
    $result = $augmentor->execute($input);
    for ($i = 1; $i <= 10; $i++) {
      $target_field_key = 'target_field_' . $i;
      $response_field_key = 'response_key_' . $i;
      if ($this->configuration[$target_field_key] && $this->configuration[$response_field_key]) {
        $target_field = $this->configuration[$target_field_key];
        $response_field = $this->configuration[$response_field_key];
        if ($object->hasField($target_field) && !empty($result) && array_key_exists($response_field, $result)) {
          $object->set($target_field, $result[$response_field]);
//          Note: We couldn't save the entity because when its already in the same presave event.
//          $object->save();
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    $fields = [];
    for ($i = 1; $i <= 10; $i++) {
      $source_field_key = 'source_field_' . $i;
      $target_field_key = 'target_field_' . $i;
      $response_field_key = 'response_key_' . $i;
      $fields[$source_field_key] = NULL;
      $fields[$target_field_key] = NULL;
      $fields[$response_field_key] = NULL;
    }
    return $fields + [
      'augmentor' => NULL,
      'action' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $entity_type = $form['type']['#value'] ?? 'node';
    $allowed_fields = $this->getAllowedFields($entity_type);
    for ($i = 1; $i <= 10; $i++) {
      $source_field_key = 'source_field_' . $i;
      $target_field_key = 'target_field_' . $i;
      $response_field_key = 'response_key_' . $i;
      $form[$source_field_key] = [
        '#type' => 'select',
        '#title' => $this->t('Source field'),
        '#options' => $allowed_fields ?? [],
        '#default_value' => $this->configuration[$source_field_key] ?? '',
        '#empty_option' => $this->t('- Select -'),
        '#required' => FALSE,
      ];
      $form[$target_field_key] = [
        '#type' => 'select',
        '#title' => $this->t('Target field'),
        '#options' => $allowed_fields ?? [],
        '#default_value' => $this->configuration[$target_field_key] ?? '',
        '#empty_option' => $this->t('- Select -'),
        '#required' => FALSE,
      ];
      $form[$response_field_key] = [
        '#type' => 'textfield',
        '#title' => $this->t('Response key to use'),
        '#default_value' => $this->configuration[$response_field_key] ?? '',
        '#empty_option' => $this->t('- Select -'),
        '#required' => FALSE,
      ];
      // Make first row of the fields mandatory.
      if ($i === 1) {
        unset($form[$source_field_key]['#empty_option']);
        $form[$source_field_key]['#required'] = TRUE;
        unset($form[$target_field_key]['#empty_option']);
        $form[$target_field_key]['#required'] = TRUE;
        unset($form[$response_field_key]['#empty_option']);
        $form[$response_field_key]['#required'] = TRUE;
        $form[$response_field_key]['#default_value'] = 'default';
      }
    }
    $form['augmentor'] = [
      '#type' => 'select',
      '#title' => $this->t('Augmentor'),
      '#options' => $this->getAugmentorOptions(),
      '#default_value' => $this->configuration['augmentor'] ?? '',
      '#required' => TRUE,
    ];
    $form['action'] = [
      '#type' => 'select',
      '#title' => $this->t('Action over target field'),
      '#options' => [
        'append' => $this->t('Append'),
        'replace' => $this->t('Replace'),
      ],
      '#default_value' => $this->configuration['action'] ?? '',
      '#required' => TRUE,
    ];
    return $form;
  }
  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    for ($i = 1; $i <= 10; $i++) {
      $source_field_key = 'source_field_' . $i;
      $target_field_key = 'target_field_' . $i;
      $response_field_key = 'response_key_' . $i;
      $this->configuration[$source_field_key] = $form_state->getValue($source_field_key);
      $this->configuration[$target_field_key] = $form_state->getValue($target_field_key);
      $this->configuration[$response_field_key] = $form_state->getValue($response_field_key);
    }
    $this->configuration['augmentor'] = $form_state->getValue('augmentor');
    $this->configuration['action'] = $form_state->getValue('action');
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, AccountInterface $account = NULL, $return_as_object = FALSE) {
    $access = AccessResult::allowed();
    return $return_as_object ? $access : $access->isAllowed();
  }

  /**
   * Returns the list of available augmentors.
   *
   * @return array
   *   The list of available augmentors.
   */
  private function getAugmentorOptions(): array {
    $options = [];
    foreach ($this->augmentorManager->getAugmentors() as $uuid => $augmentor) {
      $options[$uuid] = $augmentor['label'];
    }
    return $options;
  }

  /**
   * Returns the allowed fields for Augmentor configuration.
   *
   * @param string $entity_type
   *   The type of entity.
   *
   * @return array
   *   The list of allowed fields for Augmentor configuration.
   */
  private function getAllowedFields(string $entity_type): array {
    $fields = $this->entityFieldManager->getFieldStorageDefinitions($entity_type);
    $field_names = array_keys($fields);
    $allowed_fields = [];
    foreach ($field_names as $field_name) {
      if ($field_name == 'body' || $field_name == 'title' || str_starts_with($field_name, 'field_')) {
        $label = str_replace('field_', '', $field_name);
        $allowed_fields[$field_name] = ucwords(str_replace('_', ' ', $label));
      }
    }
    return $allowed_fields;
  }

}
