<?php

namespace Drupal\augmentor_azure_openai\Plugin\Augmentor;

use Drupal\augmentor_azure_openai\AzureOpenAIBase;
use Drupal\Core\Form\FormStateInterface;
use OpenAI\Exceptions\TransporterException;

/**
 * Azure OpenAI augmentor plugin implementation.
 *
 * @Augmentor(
 *   id = "azure_openai_chat",
 *   label = @Translation("Azure OpenAI Chat"),
 *   description = @Translation("Given a prompt, the model will return a
 *   conversational response."),
 * )
 */
class AzureOpenAIChat extends AzureOpenAIBase {

  /**
   * Available roles of the entity that is creating the message.
   */
  const MESSAGE_ROLES = [
    'user' => 'user',
    'assistant' => 'assistant',
  ];

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return parent::defaultConfiguration() + [
      'role' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['role'] = [
      '#type' => 'select',
      '#title' => $this->t('Role'),
      '#options' => self::MESSAGE_ROLES,
      '#default_value' => $this->configuration['role'] ?? 'user',
      '#description' => $this->t('The role of the entity that is creating the message. <strong>user</strong> indicates the message is sent by an actual user and should be used in most cases to represent user-generated messages. <strong>assistant</strong> indicates the message is generated by the assistant.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    $this->configuration['role'] = $form_state->getValue('role');
  }

  /**
   * Creates a completion for the provided prompt and parameters.
   *
   * @param string $input
   *   The text to use as source for the completion manipulation.
   *
   * @return array
   *   The completion output.
   */
  public function execute(string $input): array {
    try {
      $client = $this->getClient($this->configuration['base_url'], $this->configuration['api_version']);
      $preparedInput = $this->prepareText($input);
      $options = $this->buildOptions($preparedInput);
      $result = $client->chat()->create($options);
      return $this->processChoices($result->toArray());
    }
    catch (TransporterException | \Exception $e) {
      $this->logger->error('There was an issue obtaining a response from Azure OpenAI. The error was @error.', ['@error' => $e->getMessage()]);
    }
    return [];
  }

  /**
   * Builds the options array for the chat request.
   *
   * @param string $input
   *   The input string to be used in the options.
   *
   * @return array
   *   The options array for the chat request.
   */
  private function buildOptions(string $input): array {
    $role = $this->configuration['role'] ?? 'user';
    $prompt = $this->configuration['prompt'] ?? '{input}';
    $content = $role === 'user' ? str_replace('{input}', $input, $prompt) : $prompt;
    return [
      'messages' => [
        ['role' => $role, 'content' => $content],
      ],
    ];
  }

  /**
   * Processes and normalizes chat choices.
   *
   * @param array $result
   *   The result array from the chat request.
   *
   * @return array
   *   The processed and normalized output array.
   */
  private function processChoices(array $result): array {
    $output = [];

    if (isset($result['choices'])) {
      foreach ($result['choices'] as $choice) {
        $output[] = trim($this->normalizeText($choice['message']['content']), '"');
      }
    }
    return ['default' => $output];
  }

}
