<?php

namespace Drupal\augmentor_google_cloud_vision\Plugin\Augmentor;

use Drupal\augmentor_google_cloud_vision\GoogleCloudVisionBase;
use Google\Cloud\Vision\V1\ImageAnnotatorClient;

/**
 * Safe Search Detection Augmentor plugin implementation.
 *
 * @Augmentor(
 *   id = "google_cloud_vision_safe_search",
 *   label = @Translation("Google Cloud Vision Safe Search"),
 *   description = @Translation("Detects explicit content such as adult content or violent content within an image."),
 * )
 */
class SafeSearch extends GoogleCloudVisionBase {

  /**
   * Perform explicit content detection on a given image.
   *
   * @param string $path
   *   The image path to process.
   *
   * @return array
   *   Returns the likelihood of explicit content in a given image.
   *   @see https://cloud.google.com/vision/docs/detecting-safe-search
   */
  public function execute($path) {
    try {
      $this->setEnvironmentalCredentials();
      $image_annotator = new ImageAnnotatorClient();
      $response = $image_annotator->safeSearchDetection(file_get_contents($path));
      $safe_search_annotation = $response->getSafeSearchAnnotation();
      $detection_result = [];

      if ($safe_search_annotation) {
        $detection_result['adult'] = [$safe_search_annotation->getAdult()];
        $detection_result['spoof'] = [$safe_search_annotation->getSpoof()];
        $detection_result['medical'] = [$safe_search_annotation->getMedical()];
        $detection_result['violence'] = [$safe_search_annotation->getViolence()];
        $detection_result['racy'] = [$safe_search_annotation->getRacy()];
      }

      $image_annotator->close();

      return $detection_result;
    }
    catch (\Throwable $error) {
      $this->logger->error('Google Cloud Vision API error: %message.', [
        '%message' => $error->getMessage(),
      ]);
      return [
        '_errors' => $this->t('Error during the explicit content detection, please check the logs for more information.')->render(),
      ];
    }
  }

}
