<?php

namespace Drupal\authorization_code;

use Drupal\authorization_code\Exception\MissingAuthorizationMethodException;

/**
 * Class AuthorizationMethodManager
 * @package Drupal\authorization_code
 */
class AuthorizationMethodManager {

  /**
   * @var array
   */
  private static $methods;

  /**
   * Invokes the authorization_method_info hook to receive the authorization
   * method implemented by 3rd party modules.
   *
   * @return array
   *   Associative array of authorization method info.
   */
  public static function getAllAuthorizationMethods() {
    if (empty($methods)) {
      static::$methods = module_invoke_all('authorization_code_method_info');
    }
    return static::$methods;
  }

  /**
   * Returns the authorization method info of a specific id.
   *
   * @param $method_id
   *   The method id (as it is set in hook_authorization_method_info)
   * @return false|array
   *   The authorization method info details, or false if no info was found
   *   with such an id.
   */
  public static function getAuthorizationMethodInfo($method_id) {
    $methods = static::getAllAuthorizationMethods();
    if (isset($methods[$method_id])) {
      return $methods[$method_id];
    }

    return FALSE;
  }

  /**
   * Loads the user object using the provided argument.
   *
   * @param string $method_id
   *   The method id (as it is set in hook_authorization_method_info)
   * @param string $argument
   *   The authorization argument received from the client (email/phone/...).
   * @return false|object
   *   The loaded user object, or false if the authorization method callback
   *   failed to load the user.
   */
  public static function loadUserFromArgument($method_id, $argument) {
    if ($method_info = static::getAuthorizationMethodInfo($method_id)) {
      $callback = $method_info['load_user_callback'];
      if (is_callable($callback)) {
        return call_user_func_array($callback, array($argument));
      }
    }

    return FALSE;
  }

  /**
   * Validates an argument according to the method validation callback.
   *
   * @param string $method_id
   *   The method id (as it is set in hook_authorization_method_info)
   * @param mixed $argument
   *   The authorization argument received from the client (email/phone/...).
   * @return bool
   *   If the authorization method exists and the argument is valid.
   */
  public static function verifyAuthorizationArgument($method_id, $argument) {
    if ($method_info = static::getAuthorizationMethodInfo($method_id)) {
      $callback = $method_info['validate_argument_callback'];
      if (is_callable($callback)) {
        return (bool) call_user_func_array($callback, array($argument));
      }
    }

    return FALSE;
  }

  /**
   * @param string $method_id
   * @param mixed $argument
   * @param string $code
   * @throws \Drupal\authorization_code\Exception\MissingAuthorizationMethodException
   *   If the authorization method does not exist.
   */
  public static function sendAuthorizationCode($method_id, $argument, $code) {
    if ($method_info = static::getAuthorizationMethodInfo($method_id)) {
      $callback = $method_info['send_callback'];
      if (is_callable($callback)) {
        call_user_func_array($callback, array($argument, $code));
        return;
      }
    }

    throw new MissingAuthorizationMethodException();
  }

}
