<?php

namespace Drupal\authorization_code;

use Drupal\authorization_code\Exception\UpsertFailedException;

/**
 * Class AuthorizationCode
 * @package Drupal\authorization_code
 */
class AuthorizationCode {

  const TOKEN_VALUE = 'authorization_code';
  const DB_TABLE = 'authorization_code';

  /**
   * Generates an authorization code for a given key and length.
   *
   * @param string $key
   *   The authorization key to use as an index for the authorization code.
   * @param int $length
   *   The length of the authorization code.
   * @return string
   *   The generated code.
   */
  public static function generateCode($key, $length = NULL) {
    if (is_null($length)) {
      $length = AuthorizationCodeConfig::getCodeLength();
    }

    $code = static::doGenerateCode($length);
    static::upsertCode($key, $code);

    return $code;
  }

  /**
   * Generates the code.
   *
   * @param $length
   *   The length of the code.
   * @return string
   *   The generated code.
   */
  protected static function doGenerateCode($length) {
    $max = pow(10, $length);
    $random_int = mt_rand(0, $max - 1);

    // To allow preceding zeros we add $max to the value, convert it to string
    // and remove the first character.
    $value = $max + $random_int;
    $str_value = (string) $value;
    $code = substr($str_value, 1, 6);

    return $code;
  }

  /**
   * Update or inserts
   * @param $key
   * @param $code
   * @throws \Drupal\authorization_code\Exception\UpsertFailedException
   */
  protected static function upsertCode($key, $code) {
    require_once DRUPAL_ROOT . '/' . variable_get(
        'password_inc',
        'includes/password.inc'
      );

    // We use Drupal's user_hash_password to hash our authorization code.
    try {
      $record = array(
        'auth_key' => $key,
        'fetch_count' => 0,
        'code' => user_hash_password($code),
        'created' => time(),
      );
      $count_query = db_select(static::DB_TABLE);
      $count_query
        ->fields(static::DB_TABLE)
        ->condition('auth_key', $key);
      $result = $count_query->countQuery()->execute()->fetchColumn();
      $primary_keys = [];
      if ($result > 0) {
        $primary_keys = ['auth_key'];
      }

      drupal_write_record(static::DB_TABLE, $record, $primary_keys);
    }
    catch (\Exception $e) {
      throw new UpsertFailedException(
        'Failed to write auth code to the database', $e
      );
    }
  }

  /**
   * Verifies that the given code exists in the DB table and the expiration
   * time haven't passed.
   *
   * @param string $key
   *   The authorization key.
   * @param string $code
   *   The authorization code to test.
   * @return bool
   *   true if the code is valid.
   *   false otherwise.
   */
  public static function verifyCode($key, $code) {
    require_once DRUPAL_ROOT . '/' . variable_get(
        'password_inc',
        'includes/password.inc'
      );
    $query = db_select(static::DB_TABLE);
    $query
      ->fields(static::DB_TABLE, ['code'])
      ->condition('auth_key', $key)
      ->condition('created', time() - AuthorizationCodeConfig::getExpirationTime(), '>');

    if (($max_fetch_count = AuthorizationCodeConfig::getMaxFetchCount()) && ($max_fetch_count > 0)) {
      $query->condition('fetch_count', AuthorizationCodeConfig::getMaxFetchCount(), '<');
    }

    $hashed_code = $query->execute()->fetchColumn();

    // Delete all authorization codes after they are used.
    if ($hashed_code) {
      static::incrementCodeFetchCounter($key);
    }

    // We use Drupal's user_check_password to verify our hashed code.
    $fake_account = array('pass' => $hashed_code);
    return user_check_password($code, (object) $fake_account);
  }

  /**
   * Increments the fetch_count column of a provided key in the
   * authorization_code table.
   * @param string $key
   *   The authorization key.
   */
  protected static function incrementCodeFetchCounter($key) {
    db_update(static::DB_TABLE)
      ->expression('fetch_count', 'fetch_count + 1')
      ->condition('auth_key', $key)
      ->execute();
  }

}
