<?php

declare(strict_types=1);

namespace Drupal\authorization_group\Plugin\authorization\Consumer;

use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\authorization\Consumer\ConsumerPluginBase;
use Drupal\group\GroupMembershipLoaderInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Using Authorization to assign permissions to Groups.
 *
 * Groups are provided by the Group module.
 *
 * @AuthorizationConsumer(
 *   id = "authorization_group",
 *   label = @Translation("Groups")
 * )
 */
final class GroupConsumer extends ConsumerPluginBase {

  /**
   * Allow consumer target creation.
   *
   * @var bool
   */
  protected $allowConsumerTargetCreation = FALSE;

  /**
   * The membership loader service.
   *
   * @var \Drupal\group\GroupMembershipLoaderInterface
   */
  protected $membershipLoader;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Creates a group authorization consumer.
   *
   * @param array $configuration
   *   The plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param array $plugin_definition
   *   The plugin definition.
   * @param \Drupal\group\GroupMembershipLoaderInterface $membership_loader
   *   The membership loader service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    GroupMembershipLoaderInterface $membership_loader,
    EntityTypeManagerInterface $entity_type_manager,
  ) {

    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->membershipLoader = $membership_loader;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('group.membership_loader'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['delete_membership'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Remove the user from the Group, if they no longer have any roles'),
      '#default_value' => $this->configuration['delete_membership'] ?? FALSE,
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function buildRowForm(array $form, FormStateInterface $form_state, $index): array {
    $row = [];
    $mappings = $this->configuration['profile']->getConsumerMappings();

    $group_options = ['none' => $this->t('- None -')];

    $group_storage = $this->entityTypeManager->getStorage('group');

    /** @var \Drupal\group\Entity\GroupInterface[] $groups */
    $groups = $group_storage->loadMultiple();
    foreach ($groups as $group_id => $group) {
      $group_name = Html::escape($group->label());
      $group_type = $group->getGroupType();
      $group_options[$group_name][$group_id] = 'None (basic membership)';
      foreach ($group_type->getRoles() as $role) {
        if ($role->getGlobalRoleId()) {
          continue;
        }

        $group_options[$group_name][$group_id . '--' . $role->id()] = $role->label();
      }

    }
    $row['group'] = [
      '#type' => 'select',
      '#title' => $this->t('Group and Role'),
      '#options' => $group_options,
      '#default_value' => isset($mappings[$index]) ? $mappings[$index]['group'] : FALSE,
      '#description' => $this->t('Choose the Group to apply to the user.'),
    ];
    return $row;
  }

  /**
   * {@inheritdoc}
   */
  public function createConsumerTarget(string $consumer): void {
    // @todo Implement createConsumerTarget() method.
    // We are not currently allowing auto-recreation of AD groups in Drupal.
    // No work to do.
  }

  /**
   * {@inheritdoc}
   */
  public function getTokens(): array {
    $tokens = parent::getTokens();

    // Reset the tokens for plurality.
    $tokens['@' . $this->getType() . '_namePlural'] = $this->label();
    $tokens['@' . $this->getType() . '_name'] = 'Group';
    return $tokens;
  }

  /**
   * {@inheritdoc}
   */
  public function grantSingleAuthorization(UserInterface $user, $mapping, string $profile_id): void {
    $invalid_mapping = empty($mapping) || $mapping === 'none';
    $anonymous_user = $user->isAnonymous();
    $skip = $invalid_mapping || $anonymous_user;
    if ($skip) {
      return;
    }

    $group_role = explode('--', $mapping);
    $group_id = $group_role[0] ?? NULL;
    $role_id = $group_role[1] ?? NULL;

    $group_storage = $this->entityTypeManager->getStorage('group');

    /** @var \Drupal\group\Entity\GroupInterface $group */
    $group = $group_storage->load($group_id);
    $group->addMember($user);

    if (!$role_id) {
      return;
    }

    $group_role_storage = $this->entityTypeManager->getStorage('group_role');
    $role = $group_role_storage->load($role_id);
    if (!$role) {
      return;
    }

    $assign_new_role = TRUE;
    $membership_roles = [];

    // Go through all the roles that the user currently have, within the
    // group. If the user isn't assigned to the role yet, the assign it.
    // Otherwise, leave the roles as they are.
    $memberships = $group->getRelationshipsByEntity($user, 'group_membership');
    $no_memberships = empty($memberships) || !\is_array($memberships);
    if ($no_memberships) {
      return;
    }

    /** @var \Drupal\group\Entity\GroupMembership $membership */
    foreach ($memberships as $membership) {
      $group_roles = $membership->get('group_roles');
      /** @var \Drupal\Core\Field\Plugin\Field\FieldType\EntityReferenceItem $group_role */
      foreach ($group_roles as $group_role) {
        $membership_roles[] = ['target_id' => $group_role->target_id];
        if ($group_role->target_id == $role_id) {
          $assign_new_role = FALSE;
        }
      }

      if ($assign_new_role) {
        $membership_roles[] = ['target_id' => $role_id];
        $this->messenger()->addStatus($this->t('You have been granted the role %role in the %group group', [
          '%role' => $role->label(),
          '%group' => $group->label(),
        ]));
        $membership->set('group_roles', $membership_roles);
        // @todo Handle potential exception here.
        $membership->save();
      }
    }

  }

  /**
   * {@inheritdoc}
   */
  public function revokeGrants(UserInterface $user, array $context, string $profile_id): void {
    if (!$user->id()) {
      return;
    }

    // Find the granted groups and roles from the applied grants (context).
    $granted_group_roles = [];
    foreach ($context as $mapping) {
      $group_role = explode('--', $mapping);
      $group_id = $group_role[0] ?? NULL;
      $role_id = $group_role[1] ?? NULL;
      $granted_group_roles[$group_id][] = $role_id;
    }

    // Load all of the user's memberships.
    $memberships = $this->membershipLoader->loadByUser($user);

    foreach ($memberships as $membership) {

      $group = $membership->getGroup();

      // Wrapped membership relationship.
      $membership_content = $membership->getGroupRelationship();

      if (!isset($granted_group_roles[$group->id()])) {
        // Entire group was removed.
        // Either remove all roles or delete membership.
        if ($this->configuration['delete_membership']) {
          $membership_content->delete();
        }
        else {
          $membership_content->set('group_roles', []);
          $membership_content->save();
        }
      }
      else {
        // The group exists in the grants, so check all of the roles.
        $roles = $membership->getRoles();
        $roles_to_keep = [];

        foreach ($roles as $role) {
          if (\in_array($role->id(), $granted_group_roles[$group->id()])) {
            // Role exists in grants, so keep it.
            $roles_to_keep[] = ['target_id' => $role->id()];
          }
          elseif (!$role->getGlobalRoleId()) {
            $this->messenger()->addStatus($this->t('Your %role role, in the %group group, has been revoked', [
              '%role' => $role->label(),
              '%group' => $group->label(),
            ]));
          }
        }

        // Update the membership's roles.
        $membership_content->set('group_roles', $roles_to_keep);
        $membership_content->save();
      }
    }
  }

}
