<?php

namespace Drupal\webform_authorize_net_payment\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Database;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\webform_authorize_net_payment\AuthService;
use net\authorize\api\constants\ANetEnvironment;
use net\authorize\api\contract\v1\ARBCancelSubscriptionRequest;
use net\authorize\api\controller\ARBCancelSubscriptionController;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Controller to handle canceling recurring payments.
 */
class AuthorizeNetWebformCancelRecurringPaymentController extends ControllerBase {

  /**
   * The authentication service.
   *
   * @var \Drupal\webform_authorize_net_payment\AuthService
   */
  protected $authService;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new controller object.
   *
   * @param \Drupal\webform_authorize_net_payment\AuthService $authService
   *   The authentication service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service for displaying messages to the user.
   */
  public function __construct(AuthService $authService, MessengerInterface $messenger) {
    $this->authService = $authService;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
        $container->get('webform_authorize_net_payment.auth_service'),
        $container->get('messenger')
      );
  }

  /**
   * Cancels a subscription based on the provided subscription ID.
   *
   * @param string $subscriptionId
   *   The ID of the subscription to cancel.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   A response indicating the result of the cancellation process.
   */
  public function cancelSubscription($subscriptionId) {

    // Retrieve the merchant authentication details.
    $merchantAuthentication = $this->authService->getMerchantAuthentication();
    $refId = 'ref' . time();

    // Create the request to cancel the subscription.
    $request = new ARBCancelSubscriptionRequest();
    $request->setMerchantAuthentication($merchantAuthentication);
    $request->setRefId($refId);
    $request->setSubscriptionId($subscriptionId);

    // Create the controller and get the response.
    $controller = new ARBCancelSubscriptionController($request);
    $response = $controller->executeWithApiResponse(ANetEnvironment::SANDBOX);

    // Check if the response is successful.
    if ($response != NULL && $response->getMessages()->getResultCode() == 'Ok') {
      $now = new \DateTime();
      $updatedatetime = $now->getTimestamp();

      // Update the subscription status in the database.
      $connection = Database::getConnection();
      $connection->update('authorize_payment_subscription')
        ->fields([
          'status' => 'Canceled',
          'updated_date' => $updatedatetime,
        ])
        ->condition('SubscriptionId', $subscriptionId)
        ->execute();

      // Display a success message.
      $response->getMessages()->getMessage();
      $this->messenger->addStatus('Subscription ' . $subscriptionId . ' has been canceled.');

      // Redirect to the referer URL or the front page.
      $referer = $_SERVER['HTTP_REFERER'] ?? Url::fromRoute('<front>')->toString();
      $redirectResponse = new RedirectResponse($referer);
      $redirectResponse->send();
    }
    else {
      // Log the error response and display error messages.
      $errorMessages = $response->getMessages()->getMessage();
      return [
        '#markup' => $errorMessages[0]->getText() . "<a href='/admin/config/services/subscription-list'>Back to Subscription List</a>",
        '#cache' => ['max-age' => 0],
      ];
    }
  }

}
