<?php

namespace Drupal\webform_authorize_net_payment\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Database;
use Drupal\webform_authorize_net_payment\AuthService;
use net\authorize\api\constants\ANetEnvironment;
use net\authorize\api\contract\v1\ARBGetSubscriptionStatusRequest;
use net\authorize\api\controller\ARBGetSubscriptionStatusController;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller to handle subscription expiration and updates.
 */
class AuthorizeNetWebformExpireRecurringPaymentController extends ControllerBase {

  /**
   * The authentication service.
   *
   * @var \Drupal\webform_authorize_net_payment\AuthService
   */
  protected $authService;

  /**
   * Constructs a new controller object.
   *
   * @param \Drupal\webform_authorize_net_payment\AuthService $authService
   *   The authentication service.
   */
  public function __construct(AuthService $authService) {
    $this->authService = $authService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('webform_authorize_net_payment.auth_service')
    );
  }

  /**
   * Expires a subscription based on the provided subscription ID.
   *
   * @param string $subscriptionId
   *   The ID of the subscription to expire.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   A response indicating the result of the expiration process.
   */
  public function expireSubscription($subscriptionId) {

    // Retrieve merchant authentication details.
    $merchantAuthentication = $this->authService->getMerchantAuthentication();
    $refId = 'ref' . time();

    // Create the request to get subscription status.
    $request = new ARBGetSubscriptionStatusRequest();
    $request->setMerchantAuthentication($merchantAuthentication);
    $request->setRefId($refId);
    $request->setSubscriptionId($subscriptionId);

    // Execute the request and get the response.
    $controller = new ARBGetSubscriptionStatusController($request);
    $response = $controller->executeWithApiResponse(ANetEnvironment::SANDBOX);

    // Check if the response is successful.
    if ($response != NULL && $response->getMessages()->getResultCode() == 'Ok') {
      // Update the subscription status in the database.
      $now = new \DateTime();
      $updatedatetime = $now->getTimestamp();
      $expired = $response->getStatus();

      $connection = Database::getConnection();
      $connection->update('authorize_payment_subscription')
        ->fields([
          'status' => $expired,
          'updated_date' => $updatedatetime,
        ])
        ->condition('SubscriptionId', $subscriptionId)
        ->execute();

      // Return a success response.
      $responseContent = [
        'status' => 'success',
        'data' => $response,
      ];
      return new Response(json_encode($responseContent), 200, ['Content-Type' => 'application/json']);
    }
    else {
      // Return an error response if the request failed.
      $responseContent = [
        'status' => 'error',
        'message' => 'Failed to get the subscription status',
      ];
      return new Response(json_encode($responseContent), 400, ['Content-Type' => 'application/json']);
    }
  }

}
