<?php

namespace Drupal\webform_authorize_net_payment\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Database;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Url;
use Drupal\webform_authorize_net_payment\AuthService;
use net\authorize\api\constants\ANetEnvironment;
use net\authorize\api\contract\v1\CreateTransactionRequest;
use net\authorize\api\contract\v1\CreditCardType;
use net\authorize\api\contract\v1\CustomerAddressType;
use net\authorize\api\contract\v1\PaymentType;
use net\authorize\api\contract\v1\TransactionRequestType;
use net\authorize\api\controller\CreateTransactionController;
use Symfony\Component\DependencyInjection\ContainerInterface;

if (!defined('AUTHORIZENET_LOG_FILE')) {
  define('AUTHORIZENET_LOG_FILE', 'phplog');
}

/**
 * Controller for handling Authorize.Net payments.
 */
class AuthorizeNetWebformPaymentController extends ControllerBase {

  /**
   * The authentication service.
   *
   * @var \Drupal\webform_authorize_net_payment\AuthService
   */
  protected $authService;

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected $currentUser;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a new controller object.
   *
   * @param \Drupal\webform_authorize_net_payment\AuthService $authService
   *   The authentication service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user service.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service used to render HTML elements.
   */
  public function __construct(AuthService $authService, AccountProxyInterface $currentUser, RendererInterface $renderer) {
    $this->authService = $authService;
    $this->currentUser = $currentUser;
    $this->renderer = $renderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('webform_authorize_net_payment.auth_service')
    );
  }

  /**
   * Charges a credit card.
   *
   * @param array $data
   *   The payment data.
   *
   * @return array|string
   *   The payment details or an error message.
   */
  public function chargeCreditCard(array $data) {

    // Retrieve the merchant authentication details from the AuthService.
    $merchantAuthentication = $this->authService->getMerchantAuthentication();

    // Set the transaction's refId.
    $refId = 'ref' . time();

    // Create the payment data for a credit card.
    $creditCard = new CreditCardType();
    $creditCard->setCardNumber($data['credit_card_number_an']);
    $creditCard->setExpirationDate($data['credit_card_expiration_date_an']);
    $creditCard->setCardCode($data['cvv_an']);

    // Add the payment data to a paymentType object.
    $paymentType = new PaymentType();
    $paymentType->setCreditCard($creditCard);

    // Set the customer's billing address.
    $customerAddress = new CustomerAddressType();
    $customerAddress->setFirstName($data['first_name_an']);
    $customerAddress->setLastName($data['last_name_an']);
    $customerAddress->setAddress($data['address']);
    $customerAddress->setCity($data['city']);
    $customerAddress->setState($data['state']);
    $customerAddress->setZip($data['zip_code']);
    $customerAddress->setPhoneNumber($data['phone_number']);
    $customerAddress->setEmail($data['email']);

    // Create a TransactionRequestType object.
    $transactionRequestType = new TransactionRequestType();
    $transactionRequestType->setTransactionType("authCaptureTransaction");
    $transactionRequestType->setAmount($data['amount_an']);
    $transactionRequestType->setBillTo($customerAddress);
    $transactionRequestType->setPayment($paymentType);

    // Assemble the complete transaction request.
    $request = new CreateTransactionRequest();
    $request->setMerchantAuthentication($merchantAuthentication);
    $request->setRefId($refId);
    $request->setTransactionRequest($transactionRequestType);

    // Create the controller and get the response.
    $controller = new CreateTransactionController($request);
    $response = $controller->executeWithApiResponse(ANetEnvironment::SANDBOX);

    $cardDetailsWithStatus = [];

    // Check to see if the API request was successfully received and acted upon.
    if ($response !== NULL && $response->getMessages()->getResultCode() === 'Ok') {
      // The API request was successful, so check for a transaction response and
      // parse it to display the card authorization results.
      $tresponse = $response->getTransactionResponse();
      if ($tresponse !== NULL && $tresponse->getMessages() !== NULL) {
        // Transaction info.
        $transaction_id = $tresponse->getTransId();
        $payment_status = $response->getMessages()->getMessage()[0]->getText();
        $cvv = $data['cvv_an'];
        $maskedCardNumber = substr($data['credit_card_number_an'], -4);
        $cardExpiration = $data['credit_card_expiration_date_an'];
        $cardDetailsWithStatus = [
          'card_number' => $maskedCardNumber,
          'cvv' => $cvv,
          'expiration_date' => $cardExpiration,
          'transaction_id' => $transaction_id,
          'payment_status' => $payment_status,
        ];

        $now = new \DateTime();
        $paymentdate = $now->getTimestamp();
        $firstname = $data['first_name_an'];
        $lastname = $data['last_name_an'];
        $totalamount = $data['amount_an'];

        $connection = Database::getConnection();
        $connection->insert('authorize_one_time_payment')
          ->fields([
            'TransactionId' => $transaction_id,
            'FirstName' => $firstname,
            'LastName' => $lastname,
            'Amount' => $totalamount,
            'PaymentDate' => $paymentdate,
          ])
          ->execute();

        return $cardDetailsWithStatus;
      }
    }
    else {
      if (empty($response->getRefId())) {
        $roles = $this->currentUser->getRoles();
        if (in_array('administrator', $roles)) {
          $routeName = 'webform_authorize_net_payment.form';
          $url = Url::fromRoute($routeName);
          // Create a Link object using the URL and link text.
          $linkText = 'Click here';
          $link = Link::fromTextAndUrl($linkText, $url);
          // Generate the renderable link element.
          $renderableLink = $link->toRenderable();
          // Render the link element as HTML.
          $output = $this->renderer->render($renderableLink);
          return [
            '#markup' => 'Please fill authorize Configuration form fields on given link ' . $output,
          ];
        }
        else {
          return 'Payment field is Disabled. For processing payment please contact the administrator.';
        }
      }
      else {
        // Print errors if the API request wasn't successful.
        if($response->getTransactionResponse()->getErrors()) {
          return $response->getTransactionResponse()->getErrors()[0]->getErrorText();
        }
        else {
          return $response->getMessages()->getMessage()[0]->getText();
        }
      }
    }
  }

}
