<?php

namespace Drupal\webform_authorize_net_payment\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\webform_authorize_net_payment\AuthService;
use net\authorize\api\constants\ANetEnvironment;
use net\authorize\api\contract\v1\ARBGetSubscriptionRequest;
use net\authorize\api\controller\ARBGetSubscriptionController;
use Symfony\Component\DependencyInjection\ContainerInterface;

if (!defined('AUTHORIZENET_LOG_FILE')) {
  define('AUTHORIZENET_LOG_FILE', 'phplog');
}

/**
 * Controller for handling recurring payment subscriptions.
 */
class AuthorizeNetWebformPaymentGetSubscriptionController extends ControllerBase {

  /**
   * The authentication service.
   *
   * @var \Drupal\webform_authorize_net_payment\AuthService
   */
  protected $authService;

  /**
   * Constructs a new controller object.
   *
   * @param \Drupal\webform_authorize_net_payment\AuthService $authService
   *   The authentication service.
   */
  public function __construct(AuthService $authService) {
    $this->authService = $authService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('webform_authorize_net_payment.auth_service')
    );
  }

  /**
   * Creates a get subscription.
   *
   * @param string $subscriptionId
   *   The subscription ID.
   *
   * @return \net\authorize\api\contract\v1\ARBGetSubscriptionResponse
   *   The subscription response.
   */
  public function getSubscription($subscriptionId) {
    // Get the merchant authentication object from AuthService.
    $merchantAuthentication = $this->authService->getMerchantAuthentication();

    // Set the transaction's refId.
    $refId = 'ref' . time();

    // Create the API request with required parameters.
    $request = new ARBGetSubscriptionRequest();
    $request->setMerchantAuthentication($merchantAuthentication);
    $request->setRefId($refId);
    $request->setSubscriptionId($subscriptionId);
    $request->setIncludeTransactions(TRUE);

    // Create the controller.
    $controller = new ARBGetSubscriptionController($request);

    // Get the response.
    $response = $controller->executeWithApiResponse(ANetEnvironment::SANDBOX);

    if ($response !== NULL) {
      if ($response->getMessages()->getResultCode() === 'Ok') {
        $response->getMessages()->getMessage();
      }
      else {
        $response->getMessages()->getMessage();
      }
    }
    else {
      // Failed to get response.
      $this->logger('webform_authorize_net_payment')->error('Null Response Error');
    }

    return $response;
  }

}
