<?php

namespace Drupal\webform_authorize_net_payment\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Database;
use Drupal\webform_authorize_net_payment\AuthService;
use Drupal\webform_authorize_net_payment\Service\ServiceUtil;
use net\authorize\api\constants\ANetEnvironment;
use net\authorize\api\contract\v1\ARBCreateSubscriptionRequest;
use net\authorize\api\contract\v1\ARBSubscriptionType;
use net\authorize\api\contract\v1\CreditCardType;
use net\authorize\api\contract\v1\NameAndAddressType;
use net\authorize\api\contract\v1\OrderType;
use net\authorize\api\contract\v1\PaymentScheduleType;
use net\authorize\api\contract\v1\PaymentScheduleType\IntervalAType;
use net\authorize\api\contract\v1\PaymentType;
use net\authorize\api\controller\ARBCreateSubscriptionController;
use Symfony\Component\DependencyInjection\ContainerInterface;

if (!defined('AUTHORIZENET_LOG_FILE')) {
  define('AUTHORIZENET_LOG_FILE', 'phplog');
}

/**
 * Controller for handling recurring payment subscriptions.
 */
class AuthorizeNetWebformRecurringPaymentController extends ControllerBase {

  /**
   * The authentication service.
   *
   * @var \Drupal\webform_authorize_net_payment\AuthService
   */
  protected $authService;

  /**
   * Constructs a new controller object.
   *
   * @param \Drupal\webform_authorize_net_payment\AuthService $authService
   *   The authentication service.
   */
  public function __construct(AuthService $authService) {
    $this->authService = $authService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('webform_authorize_net_payment.auth_service')
    );
  }

  /**
   * Creates a new subscription.
   *
   * @param array $data
   *   The subscription data.
   *
   * @return array
   *   The subscription details with masked card information.
   */
  public function createSubscription(array $data) {
    // Retrieve the merchant authentication details from the AuthService.
    $merchantAuthentication = $this->authService->getMerchantAuthentication();
    $refId = 'ref' . time();

    // Create a new subscription type.
    $subscription = new ARBSubscriptionType();
    $subscription->setName("Donation");

    // Define the payment schedule interval.
    $interval = new IntervalAType();
    $interval->setLength($data['interval_length']);
    $interval->setUnit($data['interval_unit']);

    // Create the payment schedule with start date and total occurrences.
    $paymentSchedule = new PaymentScheduleType();
    $paymentSchedule->setInterval($interval);
    $paymentSchedule->setStartDate(new \DateTime($data['start_date']));
    $paymentSchedule->setTotalOccurrences($data['instalments']);

    // Set the subscription amount and payment schedule.
    $subscription->setPaymentSchedule($paymentSchedule);
    $subscription->setAmount($data["amount_an"]);

    // Create a credit card object with card details.
    $creditCard = new CreditCardType();
    $creditCard->setCardNumber($data['credit_card_number_an']);
    $creditCard->setExpirationDate($data['credit_card_expiration_date_an']);
    $creditCard->setCardCode($data["cvv_an"]);

    // Create a payment object using the credit card information.
    $payment = new PaymentType();
    $payment->setCreditCard($creditCard);
    $subscription->setPayment($payment);

    // Create an order object with invoice number and description.
    $order = new OrderType();
    $order->setInvoiceNumber((string) rand(1, 1000));
    $order->setDescription("Description of the subscription");
    $subscription->setOrder($order);

    // Create a billing address object.
    $billTo = new NameAndAddressType();
    $billTo->setFirstName($data['first_name_an']);
    $billTo->setLastName($data['last_name_an']);
    $billTo->setAddress($data['address']);
    $billTo->setCity($data['city']);
    $billTo->setState($data['state']);
    $billTo->setZip($data['zip_code']);
    $subscription->setBillTo($billTo);

    // Create the subscription request.
    $request = new ARBCreateSubscriptionRequest();
    $request->setmerchantAuthentication($merchantAuthentication);
    $request->setRefId($refId);
    $request->setSubscription($subscription);

    // Create subscription and get response.
    $controller = new ARBCreateSubscriptionController($request);
    $response = $controller->executeWithApiResponse(ANetEnvironment::SANDBOX);

    $cardDetailsWithStatus = [];
    if (($response != NULL) && ($response->getMessages()->getResultCode() == "Ok") && ($response->getSubscriptionId() > 0)) {
      $getSubscription = $response->getSubscriptionId();

      // Check API response for success and subscription ID.
      if ($getSubscription != NULL && $response->getMessages() != NULL) {

        // Extract and mask sensitive card information.
        $paymentStatus = $response->getMessages()->getMessage()[0]->getText();
        $cvv = $data["cvv_an"];
        $maskedCardNumber = substr($data['credit_card_number_an'], -4);
        $cvvMasked = substr_replace($cvv, str_repeat('*', strlen($cvv)), 0);
        $expirationDateMasked = substr_replace($data['credit_card_expiration_date_an'], '**', 3, 2);

        // Prepare card details with status for return.
        $cardDetailsWithStatus['card_number'] = $maskedCardNumber;
        $cardDetailsWithStatus['cvv'] = $cvvMasked;
        $cardDetailsWithStatus['expiration_date'] = $expirationDateMasked;
        $cardDetailsWithStatus['transaction_id'] = $getSubscription;
        $cardDetailsWithStatus['payment_status'] = $paymentStatus;
        // Retrieve subscription details using the getSubscription service.
        $getSubscriptionService = ServiceUtil::getRecurringPaymentService();
        $result = $getSubscriptionService->getSubscription($response->getSubscriptionId())->getSubscription();

        // Extract subscription details from the response.
        $firstName = $result->getProfile()->getPaymentProfile()->getBillTo()->getFirstName();
        $lastName = $result->getProfile()->getPaymentProfile()->getBillTo()->getLastName();
        $status = $result->getStatus();
        $totalOccurrences = $result->getPaymentSchedule()->getTotalOccurrences();
        $totalAmount = $result->getAmount();
        $startDate = $result->getPaymentSchedule()->getStartDate();
        $startDateFormatted = $startDate->getTimestamp();
        $transactionStatus = "Captured/Pending Settlement";

        // Store subscription details in the database.
        $connection = Database::getConnection();
        $connection->insert('authorize_payment_subscription')
          ->fields([
            'SubscriptionId' => $response->getSubscriptionId(),
            'FirstName' => $firstName,
            'LastName' => $lastName,
            'status' => $status,
            'NoOfOccurence' => $totalOccurrences,
            'Occurence' => '0',
            'Amount' => $totalAmount,
            'start_date' => $startDateFormatted,
            'tran_status' => $transactionStatus,
            'Updated_date' => $startDateFormatted,
          ])
          ->execute();

        // Return card details with status.
        return $cardDetailsWithStatus;
      }
    }
    else {

      // Handle API errors by returning error messages.
      $cardDetailsWithStatus = $response->getMessages()->getMessage();
      return $cardDetailsWithStatus;

    }
  }

}
