<?php

namespace Drupal\webform_authorize_net_payment\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Database;
use Drupal\webform_authorize_net_payment\AuthService;
use net\authorize\api\constants\ANetEnvironment;
use net\authorize\api\contract\v1\ARBGetSubscriptionRequest;
use net\authorize\api\controller\ARBGetSubscriptionController;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Response;

if (!defined('AUTHORIZENET_LOG_FILE')) {
  define('AUTHORIZENET_LOG_FILE', 'phplog');
}

/**
 * Controller for handling recurring payment subscriptions.
 */
class AuthorizeNetWebformRecurringPaymentGetSubscriptionController extends ControllerBase {

  /**
   * The authentication service.
   *
   * @var \Drupal\webform_authorize_net_payment\AuthService
   */
  protected $authService;

  /**
   * Constructs a new controller object.
   *
   * @param \Drupal\webform_authorize_net_payment\AuthService $authService
   *   The authentication service.
   */
  public function __construct(AuthService $authService) {
    $this->authService = $authService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('webform_authorize_net_payment.auth_service')
    );
  }

  /**
   * Gets the subscription details.
   *
   * @param string $subscriptionId
   *   The subscription ID.
   * @param string $transactionStatus
   *   The transaction status.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse|\Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function getSubscription($subscriptionId, $transactionStatus) {
    $merchantAuthentication = $this->authService->getMerchantAuthentication();
    $refId = 'ref' . time();

    $request = new ARBGetSubscriptionRequest();
    $request->setMerchantAuthentication($merchantAuthentication);
    $request->setRefId($refId);
    $request->setSubscriptionId($subscriptionId);
    $request->setIncludeTransactions(TRUE);

    $controller = new ARBGetSubscriptionController($request);
    $response = $controller->executeWithApiResponse(ANetEnvironment::SANDBOX);

    if ($response !== NULL) {
      if ($response->getMessages()->getResultCode() === 'Ok') {
        $subscription = $response->getSubscription();
        $profile = $subscription->getProfile();
        $paymentProfile = $profile->getPaymentProfile();
        $billTo = $paymentProfile->getBillTo();
        $paymentSchedule = $subscription->getPaymentSchedule();
        $arbTransactions = $subscription->getArbTransactions()[0];

        $status = $subscription->getStatus();
        $firstName = $billTo->getFirstName();
        $lastName = $billTo->getLastName();
        $totalOccurrences = $paymentSchedule->getTotalOccurrences();
        $amount = $subscription->getAmount();
        $startDate = $paymentSchedule->getStartDate()->getTimestamp();
        $updateDate = $arbTransactions->getSubmitTimeUTC()->getTimestamp();
        $numberOccurrences = $arbTransactions->getPayNum();

        $readableTransactionStatus = match ($transactionStatus) {
          'capturedPendingSettlement' => 'Captured/Pending Settlement',
          'settledSuccessfully' => $numberOccurrences . ' of ' . $totalOccurrences . ' Settled Successfully',
          default => $transactionStatus,
        };

        $connection = Database::getConnection();
        $connection->update('authorize_payment_subscription')
          ->fields([
            'status' => $status,
            'FirstName' => $firstName,
            'LastName' => $lastName,
            'NoOfOccurence' => $totalOccurrences,
            'Amount' => $amount,
            'start_date' => $startDate,
            'tran_status' => $readableTransactionStatus,
            'occurence' => $numberOccurrences,
            'updated_date' => $updateDate,
          ])
          ->condition('SubscriptionId', $subscriptionId)
          ->execute();

        $responseContent = [
          'status' => 'success',
          'data' => $subscriptionId,
        ];

        return new JsonResponse($responseContent);
      }
      else {
        $responseMessages = $response->getMessages()->getMessage();
        return new JsonResponse(['status' => 'error', 'messages' => $responseMessages], Response::HTTP_BAD_REQUEST);
      }
    }
    else {
      return new Response('No response from Authorize.Net.', Response::HTTP_BAD_REQUEST);
    }
  }

}
