<?php

namespace Drupal\webform_authorize_net_payment\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\webform_authorize_net_payment\AuthService;
use Drupal\webform_authorize_net_payment\Service\ServiceUtil;
use net\authorize\api\constants\ANetEnvironment;
use net\authorize\api\contract\v1\GetTransactionDetailsRequest as AnetAPIGetTransactionDetailsRequest;
use net\authorize\api\controller\GetTransactionDetailsController as AnetControllerGetTransactionDetailsController;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

/**
 * Handles payment capture created events.
 */
class HandlePaymentCaptureCreatedController extends ControllerBase {

  /**
   * The authentication service.
   *
   * @var \Drupal\webform_authorize_net_payment\AuthService
   */
  protected $authService;

  /**
   * Constructs a new HandlePaymentCaptureCreatedController object.
   *
   * @param \Drupal\webform_authorize_net_payment\AuthService $authService
   *   The authentication service.
   */
  public function __construct(AuthService $authService) {
    $this->authService = $authService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('webform_authorize_net_payment.auth_service'),
    );
  }

  /**
   * Handles a payment capture created event.
   *
   * @param array $payload
   *   The payload containing payment information.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response.
   */
  public function handlePaymentCaptureCreated(array $payload) {

    $id = $payload['id'];
    $merchantAuthentication = $this->authService->getMerchantAuthentication();
    $request = new AnetAPIGetTransactionDetailsRequest();

    $request->setMerchantAuthentication($merchantAuthentication);
    $request->setTransId($id);

    $controller = new AnetControllerGetTransactionDetailsController($request);
    $response = $controller->executeWithApiResponse(ANetEnvironment::SANDBOX);

    $getSubscription = $response->getTransaction()->getSubscription();
    if ($getSubscription !== NULL) {
      $subscriptionId = (string) $getSubscription->getId();
      $paymentStatus = $response->getTransaction()->getTransactionStatus();
      // $subscription =  $this->serviceUtil->getUpdateRecurringPaymentService();
      $subscription = ServiceUtil::getUpdateRecurringPaymentService();
      $subscription->getSubscription($subscriptionId, $paymentStatus);
    }

    $responseContent = [
      'status' => 'success',
      'data' => $subscriptionId ?? '',
    ];

    return new Response(json_encode($responseContent), 200, ['Content-Type' => 'application/json']);
  }

}
