<?php

namespace Drupal\auto_alter\Plugin\AutoAlterDescribeImage;

use Drupal\auto_alter\DescribeImageServiceInterface;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Link;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\file\Entity\File;
use Drupal\image\Entity\ImageStyle;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Alttext.ai integration.
 *
 * @AutoAlterDescribeImage(
 *   id = "alttext_ai",
 *   title = @Translation("Alttext.AI"),
 * )
 */
class AlttextAi implements DescribeImageServiceInterface {

  use StringTranslationTrait;

  /**
   * The file system service.
   *
   * @var Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The httpClient.
   *
   * @var GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The ConfigFactory.
   *
   * @var \Drupal\Core\Config\ConfigFactory
   */
  private $configFactory;

  /**
   * Logger Factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $loggerFactory;

  /**
   * The config object.
   *
   * @var \Drupal\Core\Config\Config|\Drupal\Core\Config\ImmutableConfig
   */
  protected $config;

  /**
   * The messenger service.
   *
   * @var \Drupal\auto_alter\Plugin\AutoAlterDescribeImage\MessengerInterface
   */
  protected $messenger;

  /**
   * Class constructor.
   */
  public function __construct(FileSystemInterface $file_system, ClientInterface $http_client, LanguageManagerInterface $language_manager, ConfigFactory $configFactory, LoggerChannelFactoryInterface $loggerFactory, MessengerInterface $messenger) {
    $this->fileSystem = $file_system;
    $this->httpClient = $http_client;
    $this->languageManager = $language_manager;
    $this->config = $configFactory->get('auto_alter.settings');
    $this->loggerFactory = $loggerFactory->get('auto_alter');
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $container->get('file_system'),
      $container->get('http_client'),
      $container->get('language_manager'),
      $container->get('config.factory'),
      $container->get('logger.factory'),
      $container->get('messenger')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function checkSetup() {
    $api_key = Xss::filter($this->config->get('api_key'));
    if (empty($api_key)) {
      return FALSE;
    }
    else {
      return TRUE;
    }
  }

  /**
   * {@inheritDoc}
   */
  public function getUri(File $file) {
    $filesize = $file->getSize();
    $uri = $file->get('uri')->value;
    if ($filesize > 1048576) {
      $style = ImageStyle::load('auto_alter_help');
      $original_uri = $uri;
      $uri = $style->buildUri($original_uri);
      $style->createDerivative($original_uri, $uri);
    }
    return $uri;
  }

  /**
   * {@inheritDoc}
   */
  public function getDescription(string $uri_or_realpath) {
    $path = $this->fileSystem->realpath($uri_or_realpath);

    // We might want to force image upload from local environments.
    if(empty($path) && !empty(getenv('ALTTEXT_AI_FORCE_IMAGE_UPLOAD'))) {
      $json = [
        'url' => \Drupal::service('file_url_generator')->generateAbsoluteString($uri_or_realpath),
      ];
    }
    else {
      $json = [
        'image' => [
          'raw' => base64_encode(file_get_contents($path)),
        ],
        'lang' => $this->languageManager->getCurrentLanguage()->getId(),
      ];
    }
    try {
      $response = $this->httpClient->post('https://alttext.ai/api/v1/images', [
        'headers' => ['X-API-Key' => $this->config->get('api_key')],
        'json' => $json,
      ]);

      if (empty($response)) {
        $this->messenger->addWarning($this->t('The Alttext.AI service returned an empty response.'));
        return '';
      }

      if ($response->getStatusCode() == 200) {
        $data = json_decode($response->getBody(), TRUE);
        if (!empty($data['alt_text'])) {
          return $data['alt_text'];
        }
      }
      else {
        $this->messenger->addWarning($this->t('The Alttext.AI service returned an error: @error', [
          '@error' => $response->getReasonPhrase(),
        ]));
      }
    }
    catch (\Exception $e) {
      $this->messenger->addWarning($this->t('The Alttext.AI service returned an error: @error', [
        '@error' => $e->getMessage(),
      ]));
    }
    return '';
  }

  public function getPluginId() {
    return 'alttext_ai';
  }

  /**
   * {@inheritDoc}
   */
  public function getPluginDefinition() {
    return [
      'id' => 'azure_cognitive_services',
      'title' => 'Azure Cognitive Services',
    ];
  }

  public function buildConfigurationForm() {
    $form = [];
    $form['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('The API key for @link', [
        '@link' => Link::fromTextAndUrl($this->t('Alttext.AI'), Url::fromUri('https://alttext.ai/'))->toString(),
      ]),
      '#default_value' => $this->config->get('api_key'),
      '#required' => TRUE,
    ];
    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function validateConfigurationForm($form_state) {
    // The API key is at least 32 characters long.
    if (strlen($form_state->getValue('api_key')) < 32) {
      $form_state->setErrorByName('api_key', $this->t('The API key is invalid.'));
    }
  }

  /**
   * {@inheritDoc}
   */
  public function submitConfigurationForm($form_state, $config) {
    $api_key = $form_state->getValue('api_key');
    $config->set('api_key', $api_key)->save();
  }

}
