<?php

declare(strict_types=1);

namespace Drupal\auto_increment_alter;

use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Site\Settings;

/**
 * Defines a base implementation that driver-specific services will extend.
 */
abstract class AutoIncrementAlter implements AutoIncrementAlterInterface {

  /**
   * Stores the name of the supported database type.
   *
   * @var string $supportedDatabaseType
   *
   * @see \Drupal\Core\Database\Connection::databaseType()
   */
  public string $supportedDatabaseType;

  /**
   * Constructs an AutoIncrementAlter object.
   */
  public function __construct(
    public readonly Connection $database,
    public readonly EntityTypeManagerInterface $entityTypeManager,
    public readonly LoggerChannelInterface $logger,
    public readonly Settings $settings,
  ) {}

  /**
   * Performs the table alter operation.
   *
   * @param string $table
   *  The name of the table to alter.
   * @param int $value
   *   The new AUTO_INCREMENT value.
   * @param bool|null $ensure_table
   *   If TRUE, verify that the table exists before trying the alter operation.
   *   Otherwise, it is assumed that the table exists. Defaults to TRUE.
   *
   * @return void
   */
  abstract protected function alterTableAutoIncrement(string $table, int $value, ?bool $ensure_table = TRUE): void;

  /**
   * {@inheritdoc}
   */
  public function alterTable(string $table, int $value): void {
    if (!$this->isDatabaseTypeSupported()) {
      return;
    }

    $this->alterTableAutoIncrement($table, $value);
  }

  /**
   * {@inheritdoc}
   */
  public function alterTables(): void {
    if (!$this->isDatabaseTypeSupported()) {
      return;
    }

    $alter_data = $this->settings->get('auto_increment_alter_tables');

    if (!is_array($alter_data)) {
      $this->logger->error('The auto_increment_alter_tables setting should be an array.');
      return;
    }

    foreach ($alter_data as $table_name => $alter_value) {
      $this->alterTableAutoIncrement($table_name, $alter_value);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function alterContentEntities(): void {
    if (!$this->isDatabaseTypeSupported()) {
      return;
    }

    $alter_data = $this->settings->get('auto_increment_alter_content_entities');

    if (!is_array($alter_data)) {
      $this->logger->error('The auto_increment_alter_content_entities setting should be an array.');
      return;
    }

    $content_entities = $this->getEntityList('content');

    // Filter out entities not present in the current installation.
    $alter_data = array_filter($alter_data, function ($entity_name) use ($content_entities) {
      $entity_exist = \array_key_exists($entity_name, $content_entities);

      if (!$entity_exist) {
        $this->logger->warning("The $entity_name entity does not exist in the current installation.");
      }

      return $entity_exist;
    }, ARRAY_FILTER_USE_KEY);

    foreach ($alter_data as $entity_name => $alter_values) {
      // Ensure array. Allows using a single integer value instead of an array.
      $alter_values = (array) $alter_values;

      // Extract alter values. If the alter value consist of only one element,
      // use it for both the base and revision tables.
      $base_table_value = $alter_values[0];
      $revision_table_value = $alter_values[1] ?? $alter_values[0];

      $base_table_name = $content_entities[$entity_name]->getBaseTable();
      $this->alterTableAutoIncrement($base_table_name, $base_table_value, FALSE);

      $revision_table_name = $content_entities[$entity_name]->getRevisionTable();
      if (!is_null($revision_table_name)) {
        $this->alterTableAutoIncrement($revision_table_name, $revision_table_value, FALSE);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityList(?string $group = NULL): array {
    $entities = $this->entityTypeManager->getDefinitions();

    return array_filter($entities, function ($entity) use ($group) {
      return !$group || $entity->getGroup() === $group;
    });
  }

  /**
   * Checks if the database type is supported by the service.
   *
   * @return bool
   *   TRUE if the database type is supported. FALSE otherwise.
   */
  public function isDatabaseTypeSupported(): bool {
    $databaseType = $this->database->databaseType();
    $isSupported = $databaseType === $this->supportedDatabaseType;

    if (!$isSupported) {
      $this->logger->error("The {$databaseType} database engine is not supported by the {$this->supportedDatabaseType} driver.");
    }

    return $isSupported;
  }

}
