<?php

declare(strict_types=1);

namespace Drupal\auto_increment_alter;

use Drupal\Core\Entity\EntityTypeInterface;

/**
 * Provides an interface for auto_increment_alter services.
 */
interface AutoIncrementAlterInterface {

  /**
   * Alters the AUTO_INCREMENT value for a single table.
   *
   * @param string $table
   *  The name of the table to alter.
   * @param int $value
   *   The new AUTO_INCREMENT value.
   *
   * @return void
   */
  public function alterTable(string $table, int $value): void;

  /**
   * Alters the AUTO_INCREMENT value for a multiple tables.
   *
   * It expects an auto_increment_alter_tables setting containing an
   * associative array of table and value pairs.
   *
   * In the auto_increment_alter_tables array the keys should be the name of the
   * table to alter. The array values should be integers containing the new
   * AUTO_INCREMENT value.
   *
   * Example:
   *
   * @code
   * $settings['auto_increment_alter_tables'] = [
   *   'node' => 500,
   *   'node_revision' => 1000,
   *   'users' => 100,
   *   'taxonomy_term_data' => 200,
   *   'taxonomy_term_revision' => 200,
   *   'file_managed' => 300,
   *   'media' => 700,
   *   'media_revision' => 700,
   * ];
   * @endcode
   *
   * @return void
   */
  public function alterTables(): void;

  /**
   * Alters the AUTO_INCREMENT value for content entities' base/revision tables.
   *
   * It expects an auto_increment_alter_content_entities setting containing an
   * associative array of table and value pairs. If a content entity is set, but
   * it does not exist in the current installation it will be filtered out
   * before calling the alter operation. The base and revision tables for the
   * content entity are identified using the 'entity_type.manager' service.
   *
   * In the auto_increment_alter_content_entities array the keys should be the
   * machine name of the content entities. The array values can be set in three
   * ways:
   *
   * 1) An array with two integer values. The first is used for the base table
   * and the second is used for the revision table.
   * 2) An array with one integer value. It is used for both the base and
   * revisions tables.
   * 3) An integer value. It is used for both the base and revisions tables.
   *
   * Some content entities do not make use of revision tables. In those cases,
   * the revision value is ignored even if set.
   *
   * Example:
   *
   * @code
   *  $settings['auto_increment_alter_content_entities'] = [
   *    'node' => [500, 1000],
   *    'user' => [100],
   *    'taxonomy_term' => [200],
   *    'file' => 300,
   *    'media' => 700,
   *  ];
   * @endcode
   *
   * @return void
   */
  public function alterContentEntities(): void;

  /**
   * Get a list of entities, optionally filtered by group.
   *
   * @param string|null $group
   *   If present only entities that belong to the specified group are returned.
   *   Possible values include 'content' and 'configuration'.
   *
   * @return EntityTypeInterface[]
   *   An array of EntityTypeInterface objects keyed by the machine name of the
   *   entities.
   *
   * @see \Drupal\Core\Entity\EntityTypeInterface::getGroup()
   */
  public function getEntityList(?string $group = NULL): array;

}
