<?php

declare(strict_types=1);

namespace Drupal\auto_increment_alter;

/**
 * Alter AUTO_INCREMENT service for MySQL.
 */
final class AutoIncrementAlterMysql extends AutoIncrementAlter {

  /**
   * {@inheritdoc}
   */
  public string $supportedDatabaseType = 'mysql';

  /**
   * {@inheritdoc}
   */
  protected function alterTableAutoIncrement($table, $value, $ensure_table = TRUE): void {
    if ($ensure_table === TRUE) {
      $exists = $this->database->schema()->tableExists($table);

      if (!$exists) {
        $this->logger->error("Cannot alter nonexistent $table table.");
        return;
      }
    }

    if ($value < 0) {
      $this->logger->error("Cannot set AUTO_INCREMENT for $table table to a negative value.");
      return;
    }

    $query = "ALTER TABLE {" . $table . "} AUTO_INCREMENT = " . $value;
    $result = $this->database->query($query)->execute();

    if ($result === TRUE) {
      $this->logger->notice("AUTO_INCREMENT value for table $table set to $value.");
    }
    else {
      $this->logger->error("It was not possible to set the AUTO_INCREMENT value for $table table.");
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getTableList(): array {
    $query = "SHOW TABLES";
    return $this->database->query($query)->fetchCol();
  }

  /**
   * {@inheritdoc}
   */
  public function getTableAutoIncrementValue($table): ?string {
    $auto_increment_values = $this->getAllTableAutoIncrementValues(FALSE);

    if (!array_key_exists($table, $auto_increment_values)) {
      $this->logger->warning("The $table table does not exists.");
      return NULL;
    }

    $table_auto_increment_value = $auto_increment_values[$table];
    if (is_null($table_auto_increment_value)) {
      $this->logger->warning("The $table table has no AUTO_INCREMENT value set.");
      return NULL;
    }

    $this->logger->notice("The AUTO_INCREMENT value of table $table is $table_auto_increment_value.");
    return $table_auto_increment_value;
  }

  /**
   * {@inheritdoc}
   */
  public function getAllTableAutoIncrementValues($filter_unset = TRUE): array {
    $query = "SELECT TABLE_NAME, AUTO_INCREMENT FROM information_schema.tables WHERE TABLE_SCHEMA = :table_schema ORDER BY TABLE_NAME";
    $arguments = [':table_schema' => $this->database->getConnectionOptions()['database']];
    $auto_increment_values = $this->database->query($query, $arguments)->fetchAllKeyed();

    if ($filter_unset === TRUE) {
      $auto_increment_values = array_filter($auto_increment_values);
    }

    return $auto_increment_values;
  }

}
