<?php

declare(strict_types=1);

namespace Drupal\auto_increment_alter\Controller;

use Drupal\auto_increment_alter\AutoIncrementAlterInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for AUTO_INCREMENT table administration.
 */
final class AutoIncrementAlterController extends ControllerBase {

  /**
   * Constructs an AutoIncrementAlterController object.
   */
  public function __construct(
    protected readonly AutoIncrementAlterInterface $autoIncrementAlterService,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('auto_increment_alter.mysql')
    );
  }

  /**
   * Lists all database tables with their AUTO_INCREMENT values.
   *
   * @return array
   *   A render array containing the table list.
   */
  public function listTables(): array {
    $autoIncrementValues = $this->autoIncrementAlterService->getTableAutoIncrementValues(TRUE);

    $header = [
      $this->t('Table name'),
      $this->t('AUTO_INCREMENT value'),
    ];

    $rows = [];
    foreach ($autoIncrementValues as $tableName => $autoIncrementValue) {
      $url = Url::fromRoute('auto_increment_alter.table', [], [
        'query' => [
          'name' => $tableName,
          'destination' => Url::fromRoute('<current>')->toString(),
        ],
      ]);
      $tableLink = [
        '#type' => 'link',
        '#title' => $tableName,
        '#url' => $url,
      ];

      $rows[] = [
        ['data' => $tableLink],
        $autoIncrementValue ?? $this->t('Not set'),
      ];
    }

    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#prefix' => '<p>' . $this->t('This list shows all tables and their AUTO_INCREMENT value.') . '</p>',
    ];
  }

}
