<?php

declare(strict_types=1);

namespace Drupal\auto_increment_alter\Form;

use Drupal\auto_increment_alter\AutoIncrementAlterInterface;
use Drupal\Component\Utility\Html;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form for altering AUTO_INCREMENT values for content entities.
 */
final class AutoIncrementAlterContentEntitiesForm extends ConfirmFormBase {

  /**
   * Constructs an AutoIncrementAlterContentEntitiesForm object.
   */
  public function __construct(
    protected readonly AutoIncrementAlterInterface $autoIncrementAlterService,
    protected readonly Settings $settings,
    protected readonly RendererInterface $renderer,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('auto_increment_alter.mysql'),
      $container->get('settings'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'auto_increment_alter_content_entities_confirm_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Do you want to alter the AUTO_INCREMENT value for the tables of multiple content entities based on the auto_increment_alter_content_entities setting?');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    $alterContentEntities = $this->settings->get('auto_increment_alter_content_entities');

    if (empty($alterContentEntities)) {
      return $this->t('The auto_increment_alter_content_entities setting is not set.');
    }

    if (!is_array($alterContentEntities)) {
      return $this->t('The auto_increment_alter_content_entities setting is not properly configured.');
    }

    $contentEntities = [];
    foreach ($alterContentEntities as $entityName => $alterValue) {
      if (is_array($alterValue)) {
        $alterValue = implode(', ', $alterValue);
      }
      $contentEntities[] = Html::escape("$entityName: $alterValue");
    }

    $description = [
      '#theme' => 'item_list',
      '#items' => $contentEntities,
      '#prefix' => '<div>' . $this->t('The tables for the following content entities will be altered:') . '</div>',
    ];

    return (string) $this->renderer->render($description);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Alter content entities');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelText() {
    return $this->t('Cancel');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('auto_increment_alter.list_tables');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);

    $alterContentEntities = $this->settings->get('auto_increment_alter_content_entities');
    if (empty($alterContentEntities) || !is_array($alterContentEntities)) {
      $form['actions']['submit']['#disabled'] = TRUE;
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->autoIncrementAlterService->alterContentEntities();
    $this->messenger()->addStatus($this->t('Tables for content entities were altered successfully.'));
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
