<?php

declare(strict_types=1);

namespace Drupal\auto_increment_alter\Form;

use Drupal\auto_increment_alter\AutoIncrementAlterInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for setting AUTO_INCREMENT value for a single content entity.
 */
final class AutoIncrementAlterContentEntityForm extends FormBase {

  /**
   * Constructs an AutoIncrementAlterContentEntityForm object.
   */
  public function __construct(
    protected readonly AutoIncrementAlterInterface $autoIncrementAlterService,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('auto_increment_alter.mysql')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'auto_increment_alter_content_entity_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $contentEntities = $this->autoIncrementAlterService->getEntityList('content');

    $entityOptions = [];
    foreach ($contentEntities as $entityId => $entityType) {
      $entityOptions[$entityId] = $entityType->getLabel();
    }

    $form['content_entity'] = [
      '#type' => 'select',
      '#title' => $this->t('Content entity'),
      '#description' => $this->t('Select a content entity to alter the AUTO_INCREMENT value for its base and revision tables'),
      '#options' => $entityOptions,
      '#required' => TRUE,
    ];

    $form['base_table_value'] = [
      '#type' => 'number',
      '#title' => $this->t('Base table AUTO_INCREMENT value'),
      '#description' => $this->t('Enter the new AUTO_INCREMENT value for the base table of the selected content entity.'),
      '#min' => 1,
      '#required' => TRUE,
    ];

    $form['revision_table_value'] = [
      '#type' => 'number',
      '#title' => $this->t('Revision table AUTO_INCREMENT value'),
      '#description' => $this->t('Enter the new AUTO_INCREMENT value for the revision table of the selected content entity. This value will only be used if the entity has a revision table.'),
      '#min' => 1,
    ];

    $form['actions'] = ['#type' => 'actions'];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $entityId = $form_state->getValue('content_entity');
    $baseTableValue = (int) $form_state->getValue('base_table_value');
    $revisionTableValue = $form_state->getValue('revision_table_value') ? (int) $form_state->getValue('revision_table_value') : NULL;

    $contentEntities = $this->autoIncrementAlterService->getEntityList('content');
    $entityLabel = $contentEntities[$entityId]->getLabel();

    $this->autoIncrementAlterService->alterContentEntity($entityId, $baseTableValue, $revisionTableValue);

    $this->messenger()->addStatus(
      $this->t('AUTO_INCREMENT value for tables of the %entity_label (%entity_id) entity have been set.', [
        '%entity_label' => $entityLabel,
        '%entity_id' => $entityId,
      ])
    );

    $this->messenger()->addStatus(
      $this->t('Base table %base_table set to %base_table_value.', [
        '%base_table' => $contentEntities[$entityId]->getBaseTable(),
        '%base_table_value' => $baseTableValue,
      ])
    );

    $revisionTable = $contentEntities[$entityId]->getRevisionTable();
    if ($revisionTable && !is_null($revisionTableValue)) {
      $this->messenger()->addStatus(
        $this->t('Revision table %revision_table set to %revision_table_value.', [
          '%revision_table' => $revisionTable,
          '%revision_table_value' => $revisionTableValue,
        ])
      );
    }
    elseif (is_null($revisionTable) && !is_null($revisionTableValue)) {
      $this->messenger()->addWarning($this->t('This entity does not have a revision table.'));
    }
  }

}
