<?php

declare(strict_types=1);

namespace Drupal\auto_increment_alter\Form;

use Drupal\Component\Utility\Html;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\Url;
use Drupal\auto_increment_alter\AutoIncrementAlterInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Confirmation form for altering AUTO_INCREMENT values for multiple tables.
 */
final class AutoIncrementAlterTablesForm extends ConfirmFormBase {

  /**
   * Constructs an AutoIncrementAlterTablesForm object.
   */
  public function __construct(
    protected readonly AutoIncrementAlterInterface $autoIncrementAlterService,
    protected readonly Settings $settings,
    protected readonly RendererInterface $renderer,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('auto_increment_alter.mysql'),
      $container->get('settings'),
      $container->get('renderer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'auto_increment_alter_tables_confirm_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion() {
    return $this->t('Do you want to alter the AUTO_INCREMENT value for multiple tables based on the auto_increment_alter_tables setting?');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    $alterTables = $this->settings->get('auto_increment_alter_tables');

    if (empty($alterTables)) {
      return $this->t('The auto_increment_alter_tables setting is not set.');
    }

    if (!is_array($alterTables)) {
      $this->t('The auto_increment_alter_tables setting is not properly configured.');
    }

    $tables = [];
    foreach ($alterTables as $tableName => $alterValue) {
      $tables[] = Html::escape("$tableName: $alterValue");
    }

    $description = [
      '#theme' => 'item_list',
      '#items' => $tables,
      '#prefix' => '<div>' . $this->t('The following tables will be altered:') . '</div>',
    ];

    return (string) $this->renderer->render($description);
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText() {
    return $this->t('Alter tables');
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);

    $alterTables = $this->settings->get('auto_increment_alter_tables');
    if (empty($alterTables) || !is_array($alterTables)) {
      $form['actions']['submit']['#disabled'] = TRUE;
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelText() {
    return $this->t('Cancel');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl() {
    return new Url('auto_increment_alter.list_tables');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->autoIncrementAlterService->alterTables();
    $this->messenger()->addStatus($this->t('Tables were altered successfully.'));
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

}
