<?php

namespace Drupal\auto_link\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;

/**
 * Plugin implementation of the 'auto_link_formatter' formatter.
 *
 * @FieldFormatter(
 *   id = "auto_link_formatter",
 *   label = @Translation("Auto Link (Convert URLs to Links)"),
 *   field_types = {
 *     "string",
 *     "text_long",
 *     "text",
 *     "string_long"
 *   }
 * )
 */
class AutoLinkFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'open_new_tab' => TRUE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = [];
    $elements['open_new_tab'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Open links in new tab'),
      '#default_value' => $this->getSetting('open_new_tab'),
    ];
    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    $summary = [];
    $summary[] = $this->getSetting('open_new_tab') ? $this->t('Opens links in a new tab.') : $this->t('Opens links in the same tab.');
    return $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      if (!empty($item->value)) {
        $text = $item->value;

        // Convert URLs into clickable links.
        $text = preg_replace_callback(
          '/(?<!["\'])(?<!\])\b(?:https?:\/\/|www\.)[^\s<]+/i',
          function ($matches) {
            $url = $matches[0];

            // Ensure "www." links have http://.
            if (strpos($url, 'www.') === 0) {
              $url = 'http://' . $url;
            }

            return '<a href="' . htmlspecialchars($url, ENT_QUOTES, 'UTF-8') . '" target="_blank" rel="noopener noreferrer">'
              . htmlspecialchars($matches[0], ENT_QUOTES, 'UTF-8') . '</a>';
          },
          $text
        );

        // Preserve new lines.
        $text = nl2br($text, FALSE);

        $elements[$delta] = [
          '#markup' => Markup::create($text),
        ];
      }
    }

    return $elements;
  }

}
