<?php

namespace Drupal\auto_node_translate\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\node\Entity\Node;

/**
 * The Translation Form.
 */
class TranslationForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'auto_node_translate_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $node = NULL) {
    $languages = \Drupal::languageManager()->getLanguages();
    $form['translate'] = [
      '#type' => 'fieldgroup',
      '#title' => t('Languages to Translate'),
      '#closed' => FALSE,
      '#tree' => TRUE,
    ];

    foreach ($languages as $language) {
      $languageId = $language->getId();
      if ($languageId !== $node->langcode->value) {
        $label = ($node->hasTranslation($languageId)) ? t('overwrite translation') : t('new translation');
        $form['translate'][$languageId] = [
          '#type' => 'checkbox',
          '#title' => t('@lang (@label)', [
            '@lang' => $language->getName(),
            '@label' => $label,
          ]),
        ];
      }
    }
    $form['save'] = [
      '#type' => 'submit',
      '#value' => $this->t('Translate'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {

  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $route = \Drupal::routeMatch();
    $node = $route->getParameter('node');
    $translations = $form_state->getValues()['translate'];
    foreach ($translations as $lid => $value) {
      if ($value) {
        $this->autoNodeTranslateNode($node, $lid);
      }
    }
    $form_state->setRedirect('entity.node.canonical', ['node' => $node->id()]);
  }

  /**
   * Translates node.
   *
   * @param \Drupal\node\Entity\Node $node
   *   The node to translate.
   * @param mixed $languageId
   *   The language id.
   */
  public function autoNodeTranslateNode(Node $node, $languageId) {
    $languageFrom = $node->langcode->value;
    $fields = $node->getFields();
    $node_trans = $this->getTransledNode($node, $languageId);
    $excludeFields = [
      'langcode',
      'default_langcode',
      'field_administrative_title',
    ];
    $config = \Drupal::config('auto_node_translate.config');
    $apiType = "Drupal\auto_node_translate\\" . $config->get('default_api');
    $api = new $apiType();
    $module_handler = \Drupal::service('module_handler');
    $module_path = $module_handler->getModule('auto_node_translate')->getPath();
    foreach ($fields as $field) {
      $fieldType = $field->getFieldDefinition()->getType();
      $translatedTypes = [
        'string',
        'string_long',
        'text',
        'text_long',
        'text_with_summary',
      ];
      $fieldName = $field->getName();
      if (in_array($fieldType, $translatedTypes)) {
        $translatedValue = [];
        $values = $field->getValue();
        foreach ($values as $key => $text) {
          if (!empty($text['value'])) {
            $translatedText = $api->translate($text['value'], $languageFrom, $languageId);
            if (in_array($fieldType, ['string', 'text']) && (strlen($translatedText) > 255)) {
              $translatedText = substr($translatedText, 0, 255);
            }
            $translatedValue[$key]['value'] = $translatedText;
            if (isset($text['format'])) {
              $translatedValue[$key]['format'] = $text['format'];
            }
          }
          else {
            $translatedValue[$key] = [];
          }
        }
        $node_trans->set($fieldName, $translatedValue);
      }
      elseif ($fieldType == 'entity_reference_revisions') {
        $targetParagraphs = $field->getValue();
        $newParagraphs = [];
        foreach ($targetParagraphs as $target) {
          $paragraph = Paragraph::load($target['target_id'], $target['target_revision_id']);
          $newParagraphs[] = $this->translateParagraph($paragraph, $api, $languageFrom, $languageId);
        }
        $node_trans->set($fieldName, $newParagraphs);
      }
      elseif (!in_array($fieldName, $excludeFields)) {
        $node_trans->set($fieldName, $node->get($fieldName)->getValue());
      }
    }
    $node->setNewRevision(TRUE);
    $node->revision_log = t('Automatica translation using @api', ['@api' => $config->get('default_api')]);
    $node->setRevisionCreationTime(REQUEST_TIME);
    $node->setRevisionUserId(\Drupal::currentUser()->id());
    $node->save();
  }

  /**
   * Translates paragraph.
   *
   * @param mixed $paragraph
   *   The paragraph to translate.
   * @param mixed $api
   *   The api to use.
   * @param mixed $languageFrom
   *   The language from.
   * @param mixed $languageId
   *   The language to.
   *
   * @return mixed
   *   fieldValue
   */
  public function translateParagraph($paragraph, $api, $languageFrom, $languageId) {
    $newParagraph = $paragraph->createDuplicate();
    $fields = $paragraph->getTranslatableFields();
    foreach ($fields as $field) {
      $fieldType = $field->getFieldDefinition()->getType();
      $translatedTypes = [
        'string',
        'string_long',
        'text',
        'text_long',
        'text_with_summary',
      ];
      if (in_array($fieldType, $translatedTypes)) {
        $translatedValue = [];
        $values = $field->getValue();
        foreach ($values as $key => $text) {
          if (!empty($text['value'])) {
            $translatedText = $api->translate($text['value'], $languageFrom, $languageId);
            if (in_array($fieldType, ['string', 'text']) && (strlen($translatedText) > 255)) {
              $translatedText = substr($translatedText, 0, 255);
            }
            if (isset($text['format'])) {
              $translatedValue[$key]['format'] = $text['format'];
            }
            $translatedValue[$key]['value'] = $translatedText;
          }
          else {
            $translatedValue[$key] = [];
          }
        }
        $fieldName = $field->getName();
        $newParagraph->set($fieldName, $translatedValue);
      }
    }
    $newParagraph->save();
    $fieldValue = [
      'target_id' => $newParagraph->id(),
      'target_revision_id' => $newParagraph->getRevisionId(),
    ];

    return $fieldValue;
  }

  /**
   * Gets or adds translated node.
   *
   * @param mixed $node
   *   The node.
   * @param mixed $languageId
   *   The language id.
   *
   * @return mixed
   *   the translated node.
   */
  public function getTransledNode(&$node, $languageId) {
    return $node->hasTranslation($languageId) ? $node->getTranslation($languageId) : $node->addTranslation($languageId);
  }

}
