<?php

declare(strict_types=1);

namespace Drupal\auto_node_translate;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Component\Datetime\TimeInterface;
use Drupal\node\Entity\Node;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\content_moderation\ModerationInformationInterface;

/**
 * Translation helpers.
 */
final class Translator {
  use StringTranslationTrait;

  /**
   * The content moderation information service.
   *
   * @var \Drupal\content_moderation\ModerationInformation|null
   */
  protected $contentModerationInformation;

  /**
   * Constructs a Translator object.
   */
  public function __construct(
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly AccountProxyInterface $currentUser,
    private readonly ConfigFactoryInterface $config,
    private readonly AutoNodeTranslateProviderPluginManager $pluginManager,
    private readonly TimeInterface $time,
  ) {}

  /**
   * Set content_moderation_information.
   *
   * @param Drupal\content_moderation\ModerationInformationInterface|null $content_moderation_information
   *   The content moderation information service.
   */
  public function setContentModerationInformation(ModerationInformationInterface|null $content_moderation_information) {
    $this->contentModerationInformation = $content_moderation_information;
  }

  /**
   * Translates node.
   *
   * @param \Drupal\node\Entity\Node $node
   *   The node to translate.
   * @param mixed $translations
   *   The translations array.
   */
  public function translateNode(Node $node, $translations) {
    $languageFrom = $node->langcode->value;
    $fields = $node->getFields();
    $excludeFields = $this->getExcludeFields();
    $translatedTypes = $this->getTextFields();
    $config = $this->config->get('auto_node_translate.settings');
    $default_api_id = $config->get('default_api');
    $api = $this->pluginManager->createInstance($default_api_id);
    foreach ($translations as $languageId => $value) {
      if ($value) {
        $node_trans = $this->getTranslatedNode($node, $languageId);
        foreach ($fields as $field) {
          $fieldType = $field->getFieldDefinition()->getType();
          $fieldName = $field->getName();
          $isTranslatable = $field->getFieldDefinition()->isTranslatable();

          if (in_array($fieldType, $translatedTypes) && !in_array($fieldName, $excludeFields)) {
            if (!$isTranslatable) {
              continue;
            }
            $translatedValue = $this->translateTextField($field, $fieldType, $api, $languageFrom, $languageId);
            $node_trans->set($fieldName, $translatedValue);
          }
          elseif ($fieldType == 'link') {
            if (!$isTranslatable) {
              continue;
            }
            $values = $this->translateLinkField($field, $api, $languageFrom, $languageId);
            $node_trans->set($fieldName, $values);
          }
          elseif ($fieldType == 'entity_reference_revisions') {
            // Process later.
          }
          elseif (!in_array($fieldName, $excludeFields)) {
            $node_trans->set($fieldName, $node->get($fieldName)->getValue());
          }
        }
        if ($this->contentModerationInformation && $this->contentModerationInformation->isModeratedEntity($node)) {
          $this->setTranslationModeratedState($node, $node_trans);
        }
      }
    }

    foreach ($fields as $field) {
      $fieldType = $field->getFieldDefinition()->getType();
      if ($fieldType == 'entity_reference_revisions') {
        $this->translateParagraphField($field, $api, $languageFrom, $translations);
      }
    }
    $node->setNewRevision(TRUE);
    $node->setRevisionLogMessage($this->t('Automatic translation using @api', ['@api' => $config->get('default_api')]));
    $node->setRevisionCreationTime($this->time->getRequestTime());
    $node->setRevisionUserId($this->currentUser->id());
    $node->save();
  }

  /**
   * Translates paragraph.
   *
   * @param mixed $paragraph
   *   The paragraph to translate.
   * @param mixed $api
   *   The api to use.
   * @param mixed $languageFrom
   *   The language from.
   * @param mixed $translations
   *   The translations array.
   */
  public function translateParagraph($paragraph, $api, $languageFrom, $translations) {
    $excludeFields = $this->getExcludeFields();
    $translatedTypes = $this->getTextFields();
    $fields = $paragraph->getFields();
    foreach ($translations as $languageId => $value) {
      if ($value) {
        $paragraph_trans = $paragraph->hasTranslation($languageId) ? $paragraph->getTranslation($languageId) : $paragraph->addTranslation($languageId);
        foreach ($fields as $field) {
          $fieldType = $field->getFieldDefinition()->getType();
          $fieldName = $field->getName();
          $isTranslatable = $field->getFieldDefinition()->isTranslatable();

          if (in_array($fieldType, $translatedTypes) && !in_array($fieldName, $excludeFields)) {
            if (!$isTranslatable) {
              continue;
            }
            $translatedValue = $this->translateTextField($field, $fieldType, $api, $languageFrom, $languageId);
            $paragraph_trans->set($fieldName, $translatedValue);
          }
          elseif ($fieldType == 'link') {
            if (!$isTranslatable) {
              continue;
            }
            $values = $this->translateLinkField($field, $api, $languageFrom, $languageId);
            $paragraph_trans->set($fieldName, $values);
          }
          elseif ($fieldType == 'entity_reference_revisions') {
            // Process later.
          }
          elseif (!in_array($fieldName, $excludeFields)) {
            $paragraph_trans->set($fieldName, $paragraph->get($fieldName)->getValue());
          }
        }
      }
    }

    foreach ($fields as $field) {
      $fieldType = $field->getFieldDefinition()->getType();
      if ($fieldType == 'entity_reference_revisions') {
        $this->translateParagraphField($field, $api, $languageFrom, $translations);
      }
    }
    $paragraph->save();
  }

  /**
   * Translates paragraph field.
   *
   * @param mixed $field
   *   The field to translate.
   * @param mixed $api
   *   The api to use.
   * @param mixed $languageFrom
   *   The language from.
   * @param mixed $translations
   *   The translations array.
   */
  public function translateParagraphField($field, $api, $languageFrom, $translations) {
    $targetParagraphs = $field->getValue();
    foreach ($targetParagraphs as $target) {
      $paragraph = $this->entityTypeManager->getStorage('paragraph')->load($target['target_id'], $target['target_revision_id']);
      $this->translateParagraph($paragraph, $api, $languageFrom, $translations);
    }
  }

  /**
   * Translates text field.
   *
   * @param mixed $field
   *   The field to translate.
   * @param string $fieldType
   *   The field type.
   * @param mixed $api
   *   The api to use.
   * @param mixed $languageFrom
   *   The language from.
   * @param mixed $languageId
   *   The language id.
   */
  public function translateTextField($field, $fieldType, $api, $languageFrom, $languageId) {
    $translatedValue = [];
    $values = $field->getValue();
    $fieldDefinition = $field->getFieldDefinition()->getFieldStorageDefinition();
    $max_length = $fieldDefinition->getSettings()['max_length'] ?? 255;

    foreach ($values as $key => $text) {
      if (!empty($text['value'])) {
        $info = [
          "field" => $field,
          "from" => $languageFrom,
          "to" => $languageId,
        ];
        $textToTranslate = $text['value'];
        $this->moduleHandler->invokeAll('auto_node_translate_translation_alter',
          [
            &$textToTranslate,
            &$info,
          ]
        );
        $translatedText = $api->translate($textToTranslate, $languageFrom, $languageId);
        if (in_array($fieldType, ['string', 'text']) && (mb_strlen($translatedText) > $max_length)) {
          $translatedText = mb_substr($translatedText, 0, $max_length);
        }
        $translatedValue[$key]['value'] = $translatedText;
        if (isset($text['format'])) {
          $translatedValue[$key]['format'] = $text['format'];
        }
      }
      else {
        $translatedValue[$key] = [];
      }
    }

    return $translatedValue;
  }

  /**
   * Translates link field.
   *
   * @param mixed $field
   *   The field to translate.
   * @param mixed $api
   *   The api to use.
   * @param mixed $languageFrom
   *   The language from.
   * @param mixed $languageId
   *   The language id.
   */
  public function translateLinkField($field, $api, $languageFrom, $languageId) {
    $values = $field->getValue();
    foreach ($values as $key => $link) {
      if (!empty($link['title'])) {
        $info = [
          "field" => $field,
          "from" => $languageFrom,
          "to" => $languageId,
        ];
        $textToTranslate = $link['title'];
        $this->moduleHandler->invokeAll('auto_node_translate_translation_alter',
        [
          &$textToTranslate,
          &$info,
        ]
        );
        $translatedText = $api->translate($textToTranslate, $languageFrom, $languageId);
        $values[$key]['title'] = $translatedText;
      }
    }
    return $values;
  }

  /**
   * Returns excluded fields.
   */
  public function getExcludeFields() : array {
    return [
      'langcode',
      'parent_id',
      'parent_type',
      'parent_field_name',
      'default_langcode',
      'id',
      'uuid',
      'revision_id',
      'type',
      'status',
      'created',
      'behavior_settings',
      'revision_default',
      'revision_translation_affected',
      'content_translation_source',
      'content_translation_outdated',
      'content_translation_changed',
    ];
  }

  /**
   * Returns text fields.
   */
  public function getTextFields() : array {
    return [
      'string',
      'string_long',
      'text',
      'text_long',
      'text_with_summary',
    ];
  }

  /**
   * Gets or adds translated node.
   *
   * @param mixed $node
   *   The node.
   * @param mixed $languageId
   *   The language id.
   *
   * @return mixed
   *   The translated node.
   */
  public function getTranslatedNode(&$node, $languageId) {
    return $node->hasTranslation($languageId) ? $node->getTranslation($languageId) : $node->addTranslation($languageId);
  }

  /**
   * Updates the moderation state of the translation.
   *
   * @param \Drupal\node\Entity\Node $node
   *   The original node.
   * @param \Drupal\node\Entity\Node $node_trans
   *   The node translation.
   */
  private function setTranslationModeratedState(Node $node, Node &$node_trans) {
    $moderation_state = $this->config->get('auto_node_translate.settings')->get('moderation_state');
    switch ($moderation_state) {
      case AutoNodeTranslateInterface::CONTENT_MODERATION_DRAFT:
        $state_id = 'draft';
        break;

      case AutoNodeTranslateInterface::CONTENT_MODERATION_PUBLISHED:
        $state_id = 'published';
        break;

      default:
        $state_id = $node->moderation_state->getString();
        break;
    }
    $workflow = $this->contentModerationInformation->getWorkflowForEntity($node);
    $state = $workflow->getTypePlugin()->getStates()[$state_id];
    $node_trans->set('moderation_state', $state_id);
    $node_trans->set('status', $state->isPublishedState());
  }

}
