<?php

declare(strict_types=1);

namespace Drupal\auto_node_translate\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\auto_node_translate\AutoNodeTranslateProviderPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\auto_node_translate\AutoNodeTranslateInterface;

/**
 * Configure Auto Node Translate settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * The plugin manager service.
   *
   * @var \Drupal\auto_node_translate\AutoNodeTranslateProviderPluginManager
   */
  protected $pluginManager;

  /**
   * If content moderation is installed on the site.
   *
   * @var bool
   */
  protected $hasContentModeration;

  /**
   * Constructs a \Drupal\auto_node_translate\Form\ConfigForm object.
   *
   * @param \Drupal\auto_node_translate\AutoNodeTranslateProviderPluginManager $plugin_manager
   *   The plugin manager service.
   * @param bool $has_content_moderation
   *   If content moderation is installed on the site.
   */
  public function __construct(AutoNodeTranslateProviderPluginManager $plugin_manager, $has_content_moderation) {
    $this->pluginManager = $plugin_manager;
    $this->hasContentModeration = $has_content_moderation;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('plugin.manager.auto_node_translate_provider'),
      $container->has('content_moderation.moderation_information'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'auto_node_translate_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['auto_node_translate.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $translators = $this->pluginManager->getDefinitions();
    $config = $this->config('auto_node_translate.settings');
    $default_api = $config->get('default_api');
    foreach ($translators as $id => $info) {
      $options[$id] = $info['label'];
    }
    $form['default_api'] = [
      '#type' => 'select',
      '#title' => $this->t('Providers'),
      '#options' => $options,
      '#default_value' => $default_api,
      '#description' => $this->t('See extra available providers <a href="https://www.drupal.org/project/auto_node_translate#providers" target="_blank">here</a>'),
    ];

    if ($this->hasContentModeration) {
      $form['moderation_state'] = [
        '#type' => 'radios',
        '#title' => $this->t('Translations moderation state'),
        '#default_value' => $config->get('moderation_state') ?? AutoNodeTranslateInterface::CONTENT_MODERATION_SAME,
        '#options' => [
          AutoNodeTranslateInterface::CONTENT_MODERATION_SAME  => $this->t('Use original node moderation state'),
          AutoNodeTranslateInterface::CONTENT_MODERATION_DRAFT => $this->t('Draft (not published)'),
          AutoNodeTranslateInterface::CONTENT_MODERATION_PUBLISHED => $this->t('Published'),
        ],
      ];
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->config('auto_node_translate.settings')
      ->set('default_api', $form_state->getValue('default_api'))
      ->set('moderation_state', $form_state->getValue('moderation_state'))
      ->save();
    parent::submitForm($form, $form_state);
  }

}
