<?php

namespace Drupal\auto_node_translate;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Url;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Description of IbmWatsonTranslationApi.
 *
 * Implementation of TranslationApiInterface for IBM Watson.
 */
class IbmWatsonTranslationApi implements TranslationApiInterface {

  use StringTranslationTrait;

  /**
   * Constructs the object.
   */
  public function __construct() {

  }

  /**
   * {@inheritdoc}
   */
  public function translate($text, $languageFrom, $languageTo) {

    $config = \Drupal::config('auto_node_translate.config');
    $from = explode('-', $languageFrom)[0];
    $to = explode('-', $languageTo)[0];
    if ($this->modelIsAvailable($from, $to)) {
      $client = \Drupal::httpClient();
      try {
        $response = $client->post($config->get('ibm_watson_url') . '/v3/translate?version=' . $config->get('ibm_watson_version'), [
          'auth' => ['apikey', $config->get('ibm_watson_apikey')],
          'json' => [
            'text' => [$text],
            'model_id' => $from . '-' . $to,
          ],
          'headers' => [
            'Content-type' => 'application/json',
          ],
        ]);
      }
      catch (BadResponseException $e) {
        $link = Url::fromRoute('auto_node_translate.config_form', [], ['absolute' => TRUE])->toString();
        \Drupal::messenger()->addStatus($this->t('Watson translator error @error .Try another Api in <a href="@link">@link</a>',
        ['@link' => $link, '@error' => $e->getMessage()]));
        return $text;
      }
      $formatedData = Json::decode($response->getBody()->getContents());
      if (isset($formatedData['translations'])) {
        $translatedText = $formatedData['translations'][0]['translation'];
      }
      else {
        $translatedText = $text;
        \Drupal::messenger()->addStatus($this->t('The translation failed for @text', ['@text' => $text]));
      }
    }
    else {
      $link = Url::fromRoute('auto_node_translate.config_form', [], ['absolute' => TRUE])->toString();
      $placeholders = [
        '@link' => $link,
        '@from' => $from,
        '@to' => $to,
      ];
      \Drupal::messenger()->addStatus($this->t("The model @from-@to isn't available in Watson. Try a different api in <a href=@link>@link</a> or try to translate form english as original language", $placeholders));
      $translatedText = $text;
    }
    \Drupal::messenger()->addStatus($this->t('translate'));
    return $translatedText;
  }

  /**
   * {@inheritdoc}
   */
  public function modelIsAvailable($languageFrom, $languageTo) {

    $config = \Drupal::config('auto_node_translate.config');
    $from = explode('-', $languageFrom)[0];
    $to = explode('-', $languageTo)[0];
    $client = \Drupal::httpClient();
    $response = $client->get($config->get('ibm_watson_url') . '/v3/models?version=' . $config->get('ibm_watson_version'), [
      'auth' => ['apikey', $config->get('ibm_watson_apikey')],
      'headers' => [
        'Content-type' => 'application/json',
      ],
    ]);
    $formatedData = Json::decode($response->getBody()->getContents());
    $models = $formatedData['models'];
    $modelExists = FALSE;
    foreach ($models as $model) {
      if ($model['model_id'] == $from . '-' . $to) {
        $modelExists = TRUE;
      }
    }
    return $modelExists;
  }

}
