<?php

declare(strict_types=1);

namespace Drupal\auto_node_translate_amazon\Plugin\AutoNodeTranslateProvider;

use Drupal\auto_node_translate\AutoNodeTranslateProviderPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Aws\Translate\TranslateClient;
use Aws\Exception\AwsException;

/**
 * Plugin implementation of the auto_node_translate_provider.
 *
 * @AutoNodeTranslateProvider(
 *   id = "auto_node_translate_amazon",
 *   label = @Translation("Amazon"),
 *   description = @Translation("Amazon translation provider for auto node translate.")
 * )
 */
final class AmazonTranslator extends AutoNodeTranslateProviderPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Aws\Translate\TranslateClient definition.
   *
   * @var Aws\Translate\TranslateClient
   */
  private $translateClient;

  /**
   * Constructs a object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config
   *   The config factory service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    ConfigFactoryInterface $config,
    MessengerInterface $messenger,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->config = $config;
    $this->messenger = $messenger;
    $config = $this->config->get('auto_node_translate_amazon.settings');
    $key = $config->get('amazon_translate_key');
    $secret = $config->get('amazon_translate_secret');
    $region = $config->get('amazon_translate_region');
    $options = [
      'version' => 'latest',
      'region'  => $region,
      'credentials' => [
        'key' => $key,
        'secret' => $secret,
      ],
    ];
    $this->translateClient = new TranslateClient($options);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('messenger'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function translate($text, $languageFrom, $languageTo) : string {
    $options = [
      'SourceLanguageCode' => $languageFrom,
      'TargetLanguageCode' => $languageTo,
      'Text' => $text,
    ];

    try {
      $result = $this->translateClient->translateText($options);
      $translation = $result['TranslatedText'];
    }
    catch (AwsException $e) {
      $this->messenger->addError($this->t('Error @code: @message', [
        '@code' => $e->getCode(),
        '@message' => $e->getMessage(),
      ]));
    }

    // Pause between translation requests to prevent
    // API flood protection.
    usleep(750);

    return (empty($translation)) ? $text : $translation;
  }

}
