<?php

declare(strict_types=1);

namespace Drupal\auto_node_translate_deepl\Plugin\AutoNodeTranslateProvider;

use Drupal\auto_node_translate\AutoNodeTranslateProviderPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Messenger\MessengerInterface;
use DeepL\Translator;

/**
 * Plugin implementation of the auto_node_translate_provider.
 *
 * @AutoNodeTranslateProvider(
 *   id = "auto_node_translate_deepl",
 *   label = @Translation("Deepl"),
 *   description = @Translation("Deepl translation provider for auto node translate.")
 * )
 */
final class DeeplTranslator extends AutoNodeTranslateProviderPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config
   *   The config factory service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    ConfigFactoryInterface $config,
    MessengerInterface $messenger,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->config = $config;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('messenger'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function translate($text, $from, $to): string {
    $config = $this->config->get('auto_node_translate_deepl.settings');
    $apiKey = $config->get('apikey');
    $translator = new Translator($apiKey);
    try {
      $result = $translator->translateText($text, $from, $to);
    }
    catch (\Exception $e) {
      $this->messenger->addError($this->t('DeeplTranslator error @error .', [
        '@error' => $e->getMessage(),
      ]));
    }
    return $result->text ?? $text;
  }

}
