<?php

declare(strict_types=1);

namespace Drupal\auto_node_translate_google\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;

/**
 * Configure Auto Node Translate Google settings for this site.
 */
final class SettingsForm extends ConfigFormBase {

  /**
   * The storage handler class for files.
   *
   * @var \Drupal\file\FileStorage
   */
  protected $fileStorage;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The form constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The Entity type manager service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The Entity type manager service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, LanguageManagerInterface $language_manager) {
    $this->fileStorage = $entity_type_manager->getStorage('file');
    $this->languageManager = $language_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('language_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'auto_node_translate_google_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['auto_node_translate_google.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('auto_node_translate_google.settings');
    $form['google_location'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Location'),
      '#description' => $this->t('Default: global'),
      '#maxlength' => 60,
      '#size' => 60,
      '#default_value' => $config->get('google_location'),
      '#required' => TRUE,
    ];

    $form['google_api']['google_api_project'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Project ID'),
      '#description' => $this->t('Your Google Cloud Platform project ID'),
      '#maxlength' => 256,
      '#size' => 128,
      '#default_value' => $config->get('google_api_project'),
      '#required' => TRUE,
    ];

    $form['google_api']['google_credentials'] = [
      '#type' => 'managed_file',
      '#title' => $this->t('Google API Credentials'),
      '#description' => $this->t('https://cloud.google.com/translate/docs/setup#php'),
      '#default_value' => $config->get('google_credentials'),
      '#upload_location' => 'private://',
      '#upload_validators' => [
        'FileExtension' => ['extensions' => 'json'],
      ],
      '#required' => TRUE,
    ];

    $form['google_api']['glossary_mappings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Glossary mappings'),
    ];
    $languages = $this->languageManager->getLanguages();
    foreach ($languages as $langcode => $language) {
      $form['google_api']['glossary_mappings']['google_api_glossary_mappings_' . $langcode] = [
        '#type' => 'textfield',
        '#title' => $this->t('Glossary for @language', ['@language' => $language->getName()]),
        '#description' => $this->t('Enter the glossary id for language: @langcode', ['@langcode' => $langcode]),
        '#default_value' => $config->get('google_api_glossary_mappings_' . $langcode) ?? NULL,
      ];
    }
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);
    $credentials = $form_state->getValues()['google_credentials'];
    if (!empty($credentials)) {
      $file = $this->fileStorage->load($credentials[0]);
      if ($file) {
        $file->setPermanent();
        $file->save();
      }
    }
    $this->config('auto_node_translate_google.settings')
      ->set('google_api_project', $form_state->getValue('google_api_project'))
      ->set('google_location', $form_state->getValue('google_location'))
      ->set('google_credentials', $credentials)
      ->save();
    $languages = $this->languageManager->getLanguages();
    foreach ($languages as $langcode => $language) {
      $this->config('auto_node_translate_google.settings')
        ->set('google_api_glossary_mappings_' . $langcode, $form_state->getValue('google_api_glossary_mappings_' . $langcode))
        ->save();
    }
  }

}
