<?php

declare(strict_types=1);

namespace Drupal\auto_node_translate_google\Plugin\AutoNodeTranslateProvider;

use Google\Cloud\Translate\V3\TranslationServiceClient;
use Google\Cloud\Translate\V3\TranslateTextGlossaryConfig;
use Drupal\auto_node_translate\AutoNodeTranslateProviderPluginBase;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Component\Utility\Html;

/**
 * Plugin implementation of the auto_node_translate_provider.
 *
 * @AutoNodeTranslateProvider(
 *   id = "auto_node_translate_google",
 *   label = @Translation("Google"),
 *   description = @Translation("Google translation provider for auto node translate.")
 * )
 */
final class GoogleTranslationApi extends AutoNodeTranslateProviderPluginBase implements ContainerFactoryPluginInterface {
  use StringTranslationTrait;

  /**
   * Google\Cloud\Translate\V3\TranslationServiceClient definition.
   *
   * @var Google\Cloud\Translate\V3\TranslationServiceClient
   */
  private $translationServiceClient;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The messenger service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   This is the service that allows us to load and save entities.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    ConfigFactoryInterface $config_factory,
    MessengerInterface $messenger,
    FileSystemInterface $file_system,
    EntityTypeManagerInterface $entity_type_manager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->config = $config_factory;
    $this->messenger = $messenger;
    $this->fileSystem = $file_system;
    $this->entityTypeManager = $entity_type_manager;
    $config = $this->config->get('auto_node_translate_google.settings');
    $credentials = $config->get('google_credentials');
    if (!empty($credentials[0])) {
      $credentialsFile = $this->entityTypeManager->getStorage('file')->load($credentials[0]);
    }
    $absolute_path = (!empty($credentialsFile)) ? $this->fileSystem->realpath($credentialsFile->getFileUri()) : NULL;
    $options = [
      'credentials' => $absolute_path,
    ];
    try {
      $this->translationServiceClient = new TranslationServiceClient($options);
    }
    catch (\Exception $th) {
      $this->messenger->addError($this->t('Error @code: @message', [
        '@code' => $th->getCode(),
        '@message' => $th->getMessage(),
      ]));
      $this->translationServiceClient = NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('messenger'),
      $container->get('file_system'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function translate($text, $languageFrom, $languageTo) : string {
    if (is_null($this->translationServiceClient)) {
      return $text;
    }
    $config = $this->config->get('auto_node_translate_google.settings');
    $projectId = $config->get('google_api_project');
    $location = $config->get('google_location');
    if (strlen($text) > 20000) {
      return $this->translate(substr($text, 0, 20000), $languageFrom, $languageTo) . $this->translate(substr($text, 20000), $languageFrom, $languageTo);
    }
    else {
      try {
        $contents = [$text];
        $targetLanguageCode = $languageTo;
        $formattedParent = $this->translationServiceClient->locationName($projectId, $location);
        $options = [
          'sourceLanguageCode' => $languageFrom,
        ];
        $glossary = $config->get('google_api_glossary_mappings_' . $languageTo);
        if (!empty($glossary)) {
          $glossaryConfig = new TranslateTextGlossaryConfig(
            [
              'glossary' => 'projects/' . $projectId . '/locations/' . $location . '/glossaries/' . $glossary,
              'ignore_case' => TRUE,
            ],
          );
          $options += [
            'glossaryConfig' => $glossaryConfig,
          ];
        }
        $response = $this->translationServiceClient->translateText($contents, $targetLanguageCode, $formattedParent, $options);
        $translation = !empty($glossary) ? $response->getGlossaryTranslations()->offsetGet(0)->getTranslatedText() :
          $response->getTranslations()->offsetGet(0)->getTranslatedText();
      }
      catch (\Exception $th) {
        $this->messenger->addError($this->t('Error @code: @message', [
          '@code' => $th->getCode(),
          '@message' => $th->getMessage(),
        ]));
      }
      finally {
        $this->translationServiceClient->close();
      }
    }
    $this->messenger->addStatus($this->t('Translation Completed'));
    return (empty($translation)) ? $text : Html::decodeEntities($translation);
  }

}
