<?php

declare(strict_types=1);

namespace Drupal\auto_node_translate_libre\Plugin\AutoNodeTranslateProvider;

use Drupal\auto_node_translate\AutoNodeTranslateProviderPluginBase;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Psr\Http\Client\ClientInterface;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Plugin implementation of the auto_node_translate_provider.
 *
 * @AutoNodeTranslateProvider(
 *   id = "auto_node_translate_libre",
 *   label = @Translation("Libre"),
 *   description = @Translation("Libre translation provider for auto node translate.")
 * )
 */
final class LibreTranslator extends AutoNodeTranslateProviderPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * The http_client service.
   *
   * @var \Psr\Http\Client\ClientInterface
   */
  protected $httpClient;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config
   *   The config factory service.
   * @param \Psr\Http\Client\ClientInterface $http_client
   *   The http_client service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    ConfigFactoryInterface $config,
    ClientInterface $http_client,
    MessengerInterface $messenger,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->config = $config;
    $this->httpClient = $http_client;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('http_client'),
      $container->get('messenger'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function translate($text, $from, $to): string {
    $config = $this->config->get('auto_node_translate_libre.settings');
    $from = explode('-', $from)[0];
    $to = explode('-', $to)[0];
    $apikey = $config->get('libretranslate_apikey');
    $url = rtrim($config->get('libretranslate_url'), '/');
    if ($this->modelIsAvailable($from, $to, $url)) {
      $client = $this->httpClient;
      try {
        $response = $client->post($url . '/translate', [
          'form_params' => [
            'api_key' => $apikey,
            'q' => $text,
            'source' => $from,
            'target' => $to,
            'format' => preg_match('#(?<=<)\w+(?=[^<]*?>)#', $text) ? 'html' : 'text',
          ],
        ]);
      }
      catch (\Exception $e) {
        $link = Url::fromRoute('auto_node_translate.config_form', [], ['absolute' => TRUE])->toString();
        $this->messenger->addError($this->t('LibreTranslate translator error @error .Try another Api in <a href="@link">@link</a>', [
          '@link' => $link,
          '@error' => $e->getMessage(),
        ]));
        return $text;
      }
      $formattedData = Json::decode($response->getBody()->getContents());
      if (isset($formattedData['translatedText'])) {
        $translatedText = $formattedData['translatedText'];
      }
      else {
        $translatedText = $text;
        $this->messenger->addStatus($this->t('The translation failed for @text', ['@text' => $text]));
      }
    }
    else {
      $link = Url::fromRoute('auto_node_translate.config_form', [], ['absolute' => TRUE])->toString();
      $placeholders = [
        '@link' => $link,
        '@from' => $from,
        '@to' => $to,
      ];
      $this->messenger->addStatus($this->t("The model @from-@to isn't available in LibreTranslate. Try a different api in <a href=@link>@link</a> or try to translate form english as original language", $placeholders));
      $translatedText = $text;
    }
    $this->messenger->addStatus($this->t('Translation Completed'));
    return $translatedText;
  }

  /**
   * {@inheritdoc}
   */
  public function modelIsAvailable($languageFrom, $languageTo, $url) {
    $client = $this->httpClient;
    try {
      $response = $client->get($url . '/languages', []);
      $languages = Json::decode($response->getBody()->getContents());
    }
    catch (\Exception $e) {
      $this->messenger->addError($this->t('LibreTranslateTranslationApi error @error .', [
        '@error' => $e->getMessage(),
      ]));
      return FALSE;
    }

    foreach ($languages as $value) {
      if ($value['code'] == $languageFrom) {
        if (in_array($languageTo, $value['targets'])) {
          return TRUE;
        }
      }
    }

    return FALSE;
  }

}
