<?php

namespace Drupal\auto_retina\Event;

use Drupal\image\ImageStyleInterface;
use Symfony\Contracts\EventDispatcher\Event;

/**
 * Defines the event for when a retina derivative is created with fewer than
 * optimum pixels.
 */
class LowQualityRetinaEvent extends Event {

  /**
   * @var array
   */
  protected $optimized;

  /**
   * @var array
   */
  protected $context;

  /**
   * @var array
   */
  protected $config;

  /**
   * LowQualityRetinaEvent constructor.
   *
   * @param $config
   * @param $context
   * @param $optimized
   */
  public function __construct($optimized, $context, $config) {
    $this->optimized = $optimized;
    $this->context = $context;
    $this->config = $config;
  }

  /**
   * Get the maximum width allowed by the source image.
   *
   * @return int
   *   Max available pixels (width).
   */
  public function getSourceWidth(): int {
    return $this->optimized['width'];
  }

  /**
   * Get the width needed for correct retina quality.
   *
   * @return int
   *   Ideal pixel width.
   */
  public function getOptimumWidth(): int {
    return $this->optimized['optimum_width'];
  }

  /**
   * Get the quality where 1 is optimum.
   *
   * @return float
   *   The lower the lesser the quality of the retina image.
   */
  public function getSubQuality() {
    return floatval($this->optimized['percent_of_optimum'] / 100);
  }

  /**
   * Get the source image URI.
   *
   * @return string
   *   The URI of the source image.
   */
  public function getSourceUri() {
    return $this->context[1];
  }

  /**
   * Get the image style being processed.
   *
   * @return \Drupal\image\ImageStyleInterface
   *   The style being used to create the derivative.
   */
  public function getImageStyle(): ImageStyleInterface {
    return $this->context[0];
  }

  /**
   * Get the retina config being used.
   *
   * @return array
   */
  public function getConfig() {
    return $this->config;
  }

}
