<?php

/**
 * @file
 * Contains \Drupal\auto_retina\Form\AutoRetinaAdminSettings.
 */

namespace Drupal\auto_retina\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RedirectDestinationInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provide an administration form for the Auto Retina module.
 */
class AutoRetinaAdminSettings extends ConfigFormBase {

  protected $moduleHandler;

  protected $destination;

  /**
   * Constructs a \Drupal\system\ConfigFormBase object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   An instance of this service.
   * @param \Drupal\Core\Routing\RedirectDestinationInterface $destination
   *   An instance of this service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, ModuleHandlerInterface $module_handler, RedirectDestinationInterface $destination) {
    parent::__construct($config_factory);
    $this->moduleHandler = $module_handler;
    $this->destination = $destination;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('module_handler'),
      $container->get('redirect.destination')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'auto_retina_admin_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['auto_retina.settings'];
  }

  public function buildForm(array $form, FormStateInterface $form_state) {
    $form['suffix'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Retina filename suffix'),
      '#description' => $this->t('This suffix appears at the end of a filename, before the dot+extension to indicate it is the retina version of an image, e.g. "@2x".  <strong>To allow more than one multiplier, add a space-separated list of suffixes, e.g. "@.75x @1.5x @2x @3x"</strong>'),
      '#default_value' => $this->config('auto_retina.settings')
        ->get('suffix'),
      '#required' => TRUE,
    ];

    if ($this->moduleHandler->moduleExists('image_style_quality')) {
      $explanation = $this->t('If a given style includes an image style quality effect it will be used as the basis, if not then the <a href=":url">image toolkit setting</a> is used.  ', [
        ':url' => Url::fromRoute('system.image_toolkit_settings', [], [
          'query' => [
            'destination' => $this->destination->get(),
          ],
        ])->toString(),
      ]);
    }
    else {
      $explanation = $this->t('The basis is the <a href=":url">image toolkit setting</a>. <em>(Consider installing the <a href=":module" target="_blank">Image Style Quality</a> module for more options.)</em>  ', [
        ':url' => Url::fromRoute('system.image_toolkit_settings', [], [
          'query' => [
            'destination' => $this->destination->get(),
          ],
        ])->toString(),
        ':module' => Url::fromUri('https://www.drupal.org/project/image_style_quality')
          ->toString(),
      ]);
    }

    $form['quality_multiplier'] = [
      '#title' => $this->t('JPEG Quality Multiplier (for magnified images only)'),
      '#type' => 'number',
      '#step' => '.05',
      '#min' => '.05',
      '#max' => '1',
      '#field_suffix' => 'x JPEG percentage',
      '#size' => 4,
      '#description' => $this->t('Affect the JPEG quality used when generating the magnified image(s).  @explanation<strong>If the basis is 80% and you leave this at 1, the retina image will be generated at 80% (no change); however if you set this to .5 then the retina image will be generated at 40%.</strong>  The lower the percentage multiplier, the smaller will be the file size of the retina image, <em>at the expense of quality</em>.', [
        '@explanation' => $explanation,
      ]),
      '#default_value' => $this->config('auto_retina.settings')
        ->get('quality_multiplier'),
    ];


    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced'),
      '#open' => FALSE,
    ];

    $form['advanced']['regex'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Retina filename regex'),
      '#description' => $this->t('Enter a regex expression to use for determining if an url is retina.  The token <code>SUFFIX</code> may be used to dynamically populate the setting from above. You may omit start/end delimiters.'),
      '#default_value' => $this->config('auto_retina.settings')
        ->get('regex'),
      '#required' => TRUE,
    ];

    $form['advanced']['js'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include the javascript settings <code>drupalSettings.autoRetina</code> on every page?'),
      '#default_value' => $this->config('auto_retina.settings')
        ->get('js'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('auto_retina.settings');
    $config->set('suffix', $form_state->getValue($form['suffix']['#parents']));
    if (isset($form['quality_multiplier'])) {
      $config->set('quality_multiplier', $form_state->getValue($form['quality_multiplier']['#parents']));
    }
    $config->set('regex', $form_state->getValue($form['advanced']['regex']['#parents']));
    $config->set('js', $form_state->getValue($form['advanced']['js']['#parents']));
    $config->save();
    parent::submitForm($form, $form_state);
  }
}
