<?php

namespace Drupal\auto_translation\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\State\StateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandler;

/**
 * Form with auto_translation on how to use cron.
 */
class AutoTranslatorSettingsForm extends ConfigFormBase {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The state keyvalue collection.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected $state;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Module handler service object.
   *
   * @var \Drupal\Core\Extension\ModuleHandler
   */
  protected $moduleHandler;

  /**
   * {@inheritdoc}
   */
  public function __construct(ConfigFactoryInterface $config_factory, AccountInterface $current_user, StateInterface $state, EntityTypeManagerInterface $entity_type_manager, ModuleHandler $moduleHandler) {
    parent::__construct($config_factory);
    $this->currentUser = $current_user;
    $this->state = $state;
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleHandler = $moduleHandler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $form = new static(
        $container->get('config.factory'), $container->get('current_user'), $container->get('state'), $container->get('entity_type.manager'), $container->get('module_handler')
    );
    $form->setMessenger($container->get('messenger'));
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'auto_translation';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->configFactory->get('auto_translation.settings');
    $node_options = [];
    $node_options_default = [];
    $nodes_type = $this->entityTypeManager
      ->getStorage('node_type')
      ->loadMultiple();

    foreach ($nodes_type as $type) {
      $node_options[$type->id()] = $type->label();
      $node_options_default[$type->id()] = $type->id();
    }

    if (!empty($config->get('auto_translation_content_types'))) {
      $enabled_content_type = $config->get('auto_translation_content_types');
    }
    else {
      $enabled_content_type = $node_options_default;
    }
    $form['configuration_nodes'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Auto Translation - configuration'),
      '#open' => TRUE,
    ];
    $form['configuration_nodes']['content_types_list'] = [
      '#type' => 'details',
      '#title' => $this->t('Auto Translation - Content types configuration'),
      '#open' => TRUE,
    ];
    $form['configuration_nodes']['api_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Auto Translation - API configuration'),
      '#open' => TRUE,
  ];
    $form['configuration_nodes']['intro'] = [
      '#type' => 'item',
      '#markup' => $this->t('You can select content types where enable the auto translation, by default is enabled on all content types with Google Tranlsate browser Free API.'),
      '#weight' => -10,
    ];
    $form['configuration_nodes']['content_types_list']['auto_translation_content_types'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Enabled Content Types'),
      '#description' => $this->t('Define what content types will be enabled for content auto translation.'),
      '#default_value' => $enabled_content_type,
      '#options' => $node_options,
    ];
    $form['configuration_nodes']['api_settings']['auto_translation_api_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable API'),
      '#description' => $this->t('Enable server side API for content auto translation, if unchecked Google Translate browser Free API will be used.'),
      '#default_value' => $config->get('auto_translation_api_enabled'),
    ];
    $form['configuration_nodes']['api_settings']['auto_translation_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('Enter your Google API key.'),
      '#default_value' => $config->get('auto_translation_api_key'),
      '#states' => [
        'visible' => [
          ':input[name="auto_translation_api_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];
    $form['configuration_nodes']['api_settings']['auto_translation_api_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Secret'),
      '#description' => $this->t('Enter your Google API secret.'),
      '#default_value' => $config->get('auto_translation_api_secret'),
      '#states' => [
        'visible' => [
          ':input[name="auto_translation_api_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];
    return parent::buildForm($form, $form_state);
  }


  /**
   * Add the items to the queue when signaled by the form.
   */
  public function addItems(array &$form, FormStateInterface &$form_state) {
    $values = $form_state->getValues();
    $queue_name = $form['cron_queue_setup']['queue'][$values['queue']]['#title'];
    $num_items = $form_state->getValue('num_items');
    // Queues are defined by a QueueWorker Plugin which are selected by their
    // id attritbute.
    // @see \Drupal\auto_translation\Plugin\QueueWorker\ReportWorkerOne
    $queue = $this->queue->get($values['queue']);

    for ($i = 1; $i <= $num_items; $i++) {
      // Create a new item, a new data object, which is passed to the
      // QueueWorker's processItem() method.
      $item = new \stdClass();
      $item->created = REQUEST_TIME;
      $item->sequence = $i;
      $queue->createItem($item);
    }

    $args = [
      '%num' => $num_items,
      '%queue' => $queue_name,
    ];
    $this->messenger()->addMessage($this->t('Added %num items to %queue', $args));
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Update the interval as stored in configuration. This will be read when
    // this modules hook_cron function fires and will be used to ensure that
    // action is taken only after the appropiate time has elapsed.
    $this->configFactory->getEditable('auto_translation.settings')
      ->set('interval', $form_state->getValue('auto_translation_interval'))
      ->set('auto_translation_content_types', $form_state->getValue('auto_translation_content_types'))
      ->set('auto_translation_moderation_state_settings', $form_state->getValue('auto_translation_moderation_state_settings'))
      ->set('auto_translation_enablemoderation', $form_state->getValue('auto_translation_enablemoderation'))
      ->save();

    parent::submitForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['auto_translation.settings'];
  }

}
