<?php

namespace Drupal\auto_translation;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Google\Cloud\Translate\V2\TranslateClient;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;

/**
 * Utility class for auto_translation module functions.
 *
 * @package Drupal\auto_translation
 */
class Utility {

  use StringTranslationTrait;
  use LoggerChannelTrait;

  /**
   * Translates the given text from the source language to the target language.
   *
   * @param string $text
   *   The text to be translated.
   * @param string $s_lang
   *   The source language code.
   * @param string $t_lang
   *   The target language code.
   *
   * @return string
   *   The translated text.
   */
  public function translate($text, $s_lang, $t_lang) {
    $config = $this->config();
    $translation = NULL;
    $provider = $config->get('auto_translation_provider') ? $config->get('auto_translation_provider') : 'google';
    $serverApiEnalbled = $config->get('auto_translation_api_enabled') ? $config->get('auto_translation_api_enabled') : NULL;
    if ($provider == 'google') {
      if ($serverApiEnalbled) {
        $translation = $this->translateApiServerCall($text, $s_lang, $t_lang);
      } else {
        $text = rawurlencode($text);
        $translation = $this->translateApiBrowserCall($text, $s_lang, $t_lang);
      }
    } else if ($provider == 'libretranslate') {
      $translation = $this->libreTranslateApiCall($text, $s_lang, $t_lang);
    }
    return $translation;
  }
  /**
   * Translates the given text using the API libre translate server.
   *
   * @param string $text
   *   The text to be translated.
   * @param string $s_lang
   *   The source language of the text.
   * @param string $t_lang
   *   The target language for the translation.
   *
   * @return string
   *   The translated text.
   */
  public function libreTranslateApiCall($text, $s_lang, $t_lang) {
    $config = $this->config();
    $client = new Client();
    $logger = $this->getLogger('auto_translation');
    $translation = NULL;
    $endpoint = 'https://libretranslate.com/translate';
    $httpclient_body = [
      'q' => $text,
      'source' => $s_lang,
      'target' => $t_lang,
      'format' => "text",
      'api_key' => $config->get('auto_translation_api_key')

    ];
    $header = [
      'Content-Type' => "application/json"
    ];
    $http_cleint_options = [
      'headers' => $header,
      'body' => json_encode($httpclient_body),
      'verify' => FALSE,
    ];
    try {
      $response = $client->post($endpoint, $http_cleint_options);

      $data = $response->getBody()->getContents();
      $result = Json::decode($data);
      $translation = isset($result['translatedText']) && !empty($result['translatedText']) ? $$result['translatedText'] : NULL;
    } catch (RequestException $exception) {
      $output = urldecode($exception->getMessage());
      $to_remove = $this->getStringBetween($output, 'error: `', '` ');
      $path_remove = explode("response:", str_replace($to_remove, '', $output));
      $outputUpdated = $path_remove[0];
      $message = str_replace('`', '', $path_remove[0]);
      $error_code = $this->getStringBetween($output, 'a `', ' ');
      $reason = $this->getStringBetween($outputUpdated, $error_code, '`');

      $error = [
        'Error code' => $error_code ? $error_code : '400',
        'Error message' => $message ? $message : 'default error message',
        'Error reason' => $reason ? strtolower($reason) : 'default error reason',
      ];
      $logger->error('Auto translation error: @error', ['@error' => json_encode($error)]);
    }
    return $translation;
  }

  /**
   * Translates the given text using the API server.
   *
   * @param string $text
   *   The text to be translated.
   * @param string $s_lang
   *   The source language of the text.
   * @param string $t_lang
   *   The target language for the translation.
   *
   * @return string
   *   The translated text.
   */
  public function translateApiServerCall($text, $s_lang, $t_lang) {
    $config = $this->config();
    $client = new TranslateClient([
      'key' => $config->get('auto_translation_api_key'),
    ]);
    $translation = NULL;
    try {
      $result = $client->translate($text, [
        'source' => $s_lang,
        'target' => $t_lang,
      ]);
      $translation = htmlspecialchars_decode($result['text']);
    } catch (RequestException $e) {
      $this->getLogger('auto_translation')->error('Auto translation error: @error', ['@error' => $e->getMessage()]);
    }
    return $translation;
  }

  /**
   * Translates the given text using the API browser call.
   *
   * @param string $text
   *   The text to be translated.
   * @param string $s_lang
   *   The source language of the text.
   * @param string $t_lang
   *   The target language for the translation.
   *
   * @return string
   *   The translated text.
   */
  public function translateApiBrowserCall($text, $s_lang, $t_lang) {
    $client = new Client();
    $logger = $this->getLogger('auto_translation');
    $httpclient_options = [];
    $httpclient_options['verify'] = FALSE;
    $httpclient_options['cache-control'] = 'no-cache';
    $translation = NULL;
    $googleEndpoint = 'https://translate.googleapis.com/translate_a/single?client=gtx&sl=';
    try {
      $response = $client->get($googleEndpoint . $s_lang . '&tl=' . $t_lang . '&dt=t&q=' . $text, $httpclient_options);
      $data = $response->getBody()->getContents();
      $result = Json::decode($data);
      $string = '';
      if (isset($result[0]) && !empty($result[0])) {
        foreach ($result[0] as $key => $value) {
          if ($value !== NULL && $value[0] && !empty($value[0])) {
            $string .= $value[0];
          }
        }
      }
      $translation = $string && !empty($string) && $string !== '' ? $string : NULL;
    } catch (RequestException $exception) {
      $output = urldecode($exception->getMessage());
      $to_remove = $this->getStringBetween($output, 'error: `', '` ');
      $path_remove = explode("response:", str_replace($to_remove, '', $output));
      $outputUpdated = $path_remove[0];
      $message = str_replace('`', '', $path_remove[0]);
      $error_code = $this->getStringBetween($output, 'a `', ' ');
      $reason = $this->getStringBetween($outputUpdated, $error_code, '`');

      $error = [
        'Error code' => $error_code ? $error_code : '400',
        'Error message' => $message ? $message : 'default error message',
        'Error reason' => $reason ? strtolower($reason) : 'default error reason',
      ];
      $logger->error('Auto translation error: @error', ['@error' => json_encode($error)]);
    }
    return $translation;
  }

  /**
   * Custom function to return saved resources.
   */
  public function getEnabledContentTypes() {
    $config = $this->config();
    $enabledContentTypes = $config->get('auto_translation_content_types') ? $config->get('auto_translation_content_types') : NULL;
    return $enabledContentTypes;
  }

  /**
   * Retrieves the excluded fields.
   *
   * @return array
   *   The excluded fields.
   */
  public function getExcludedFields() {
    $config = $this->config();
    $excludedFields = [
      'und',
      'published',
      'unpublished',
      '0',
      '1',
      'draft',
      'ready for review',
      'language',
      'boolean',
      'created',
      'changed',
      'datetime',
      'path',
      NULL,
    ];
    $excludedFieldsSettings = $config->get('auto_translation_excluded_fields') ? $config->get('auto_translation_excluded_fields') : NULL;
    if ($excludedFieldsSettings) {
      $excludedFieldsSettings = explode(",", $excludedFieldsSettings);
      $excludedFields = array_merge($excludedFields, $excludedFieldsSettings);
    }
    return $excludedFields;
  }
  /**
   * Implements auto translation form function.
   */
  function formTranslate($form, $form_state) {
    $current_path = \Drupal::service('path.current')->getPath();
    $enabledContentTypes = $this->getEnabledContentTypes();
    if ($form_state->getFormObject() instanceof EntityForm) {
      $entity = $form_state->getFormObject()->getEntity();
    } else {
      $entity = $form_state->getFormObject()->entity;
    }
    $entity_type = $entity->getEntityTypeId();
    $bundle = $entity->bundle();
    $arrCheck = $this->getExcludedFields();
    if (strpos($current_path, 'translations/add') !== FALSE && in_array($bundle, $enabledContentTypes)) {
      $fields = \Drupal::service('entity_field.manager')->getFieldDefinitions($entity_type, $bundle);
      $t_lang = $entity->langcode->value;
      $d_lang = \Drupal::service('language_manager')->getDefaultLanguage()->getId();

      foreach ($fields as $field) {
        $field_name = $field->getName();
        $field_type = $field->getType();

        // Transletable field support.
        if ($field->isTranslatable()) {
          // Translate field.
          if (
            is_string($entity->get($field_name)->value)
            && !in_array(strtolower($entity->get($field_name)->value), $arrCheck)
            && $field_name != "langcode"
            && !is_numeric($entity->get($field_name)->value)
            && !in_array($field_type, $arrCheck)
            && isset($form[$field_name]['widget'][0]['value']['#default_value'])
          ) {
            $string = $entity->get($field_name)->value ? (string) $entity->get($field_name)->value : '';
            if ($string && !empty($string) && $string !== '' && strip_tags($string) !== '' && !empty(strip_tags($string))) {
              $translationResponse = $this->translate($string, $d_lang, $t_lang);
              if ($translationResponse) {
                $form[$field_name]['widget'][0]['value']['#default_value'] = $translationResponse;
              }
            }
          }
          // Paragraphs reference field support.
          if ($field->getSetting('target_type') == 'paragraph') {
            // Translate field.
            if (!in_array($field_type, $arrCheck)) {
              $paragraphs = $entity->get($field_name)->getValue();
              foreach ($paragraphs as $paragraph) {
                foreach ($paragraph as $key => $value) {
                  if (
                    is_string($value) && !in_array(strtolower($value), $arrCheck) && $key != "langcode" && !is_numeric($value) && !in_array($field_type, $arrCheck)
                    && strip_tags($value) !== '' && !empty(strip_tags($value))
                  ) {
                    $translationResponse = $this->translate($value, $d_lang, $t_lang);
                    if ($translationResponse) {
                      $form[$field_name]['widget'][0][$key]['#default_value'] = $translationResponse;
                    }
                  }
                }
              }
            }
          }
          if (isset($form[$field_name]['widget'][0]['summary']["#default_value"]) && is_string($form[$field_name]['widget'][0]['summary']["#default_value"]) && !in_array('summary', $arrCheck)) {
            $string = $form[$field_name]['widget'][0]['summary']["#default_value"] ? $form[$field_name]['widget'][0]['summary']["#default_value"] : '';
            if ($string && !empty($string) && $string !== '') {
              $translationResponse = $this->translate($string, $d_lang, $t_lang);
              if ($translationResponse) {
                $form[$field_name]['widget'][0]['summary']["#default_value"] = $translationResponse;
              }
            }
          }
          if (isset($form[$field_name]['widget'][0]["#default_value"]) && is_string($form[$field_name]['widget'][0]["#default_value"]) && !in_array($field_name, $arrCheck)) {
            $string = $form[$field_name]['widget'][0]["#default_value"] ? $form[$field_name]['widget'][0]["#default_value"] : '';
            if ($string && !empty($string) && $string !== '' && strip_tags($string) !== '' && !empty(strip_tags($string))) {
              $translationResponse = $this->translate($string, $d_lang, $t_lang);
              if ($translationResponse) {
                $form[$field_name]['widget'][0]["#default_value"] = $translationResponse;
              }
            }
          }
        }
      }
    }
    return $form;
  }
  /**
   * Custom get string beetwin function.
   */
  public function getStringBetween($string, $start, $end) {
    $string = ' ' . $string;
    $ini = strpos($string, $start);
    if ($ini == 0) {
      return '';
    }
    $ini += strlen($start);
    $len = strpos($string, $end, $ini) - $ini;
    return substr($string, $ini, $len);
  }

  /**
   * Retrieves the container.
   *
   * @return mixed
   *   The container.
   */
  public static function getContainer() {
    return \Drupal::getContainer();
  }

  /**
   * Retrieves the configuration settings.
   *
   * @return object
   *   The configuration settings.
   */
  public static function config() {
    return static::getContainer()
      ->get('config.factory')
      ->get('auto_translation.settings');
  }

  /**
   * Returns the path of the module.
   *
   * @return string
   *   The path of the module.
   */
  public static function getModulePath() {
    return static::getContainer()
      ->get('extension.list.module')
      ->getPath('auto_translation');
  }

  /**
   * Retrieves the specified module by name.
   *
   * @param string $module_name
   *   The name of the module to retrieve.
   *
   * @return mixed|null
   *   The module object if found, null otherwise.
   */
  public static function getModule($module_name) {
    return static::getContainer()
      ->get('extension.list.module')
      ->get($module_name);
  }
}
