<?php

/**
 * @file
 * Configuration for autoban module.
 */

/**
 * Menu callback. Displays autoban rules list.
 */
function autoban_page() {
  $rows = array();
  $header = array(
    array('data' => t('ID'), 'field' => 'rid', 'sort' => 'desc'),
    array('data' => t('Type'), 'field' => 'type'),
    array('data' => t('Message pattern'), 'field' => 'message'),
    array('data' => t('Threshold'), 'field' => 'threshold'),
    array('data' => t('User type'), 'field' => 'user_type'),
    array('data' => t('IP type'), 'field' => 'ip_type'),
    t('Actions'),
  );
  $build = array();

  $autoban_list = autoban_get_rules(NULL, $header);
  if (!empty($autoban_list)) {
    // All rules actions button.
    $build['autoban_ban_all_form'] = drupal_get_form('autoban_ban_all_form');
    foreach ($autoban_list as $rule) {
      $rows[] = array(
        $rule->rid,
        filter_xss($rule->type),
        filter_xss($rule->message),
        $rule->threshold,
        _autoban_get_user_type_name($rule->user_type),
        _autoban_get_ip_type_name($rule->ip_type),
        l(t('edit'), AUTOBAN_BASE_URL . "/edit/$rule->rid")
        . ' ' . l(t('delete'), AUTOBAN_BASE_URL . "/delete/$rule->rid")
        . ' ' . l(t('test'), AUTOBAN_BASE_URL . "/test/$rule->rid"),
      );
    }
  }

  $build['autoban_form'] = drupal_get_form('autoban_form');
  $build['autoban_table'] = array(
    '#theme' => 'table',
    '#header' => $header,
    '#rows' => $rows,
    '#empty' => t('Table has no rows!'),
  );
  return $build;
}

/**
 * Menu callback. Test autoban rule page.
 *
 * @param int $rid
 *   Autoban rule id.
 */
function autoban_test($rid) {
  $build = array();
  $build['backlink'] = array(
    '#type' => 'markup',
    '#markup' => l(t('Back'), AUTOBAN_BASE_URL),
  );

  $rule = autoban_get_rules($rid);
  if (!empty($rule)) {
    $params = t("Parameters: Type='%type' Message like '%message' Threshold>=%threshold User type=%user_type IP type=%ip_type.",
      array(
        '%type' => filter_xss($rule->type),
        '%message' => filter_xss($rule->message),
        '%threshold' => $rule->threshold,
        '%user_type' => _autoban_get_user_type_name($rule->user_type),
        '%ip_type' => _autoban_get_ip_type_name($rule->ip_type),
      ));

    $build['parameters'] = array(
      '#type' => 'markup',
      '#markup' => '<h2>' . $params . '</h2>',
    );

    $rows = array();
    $header = array(
      '#',
      array('data' => t('Hostname'), 'field' => 'hostname'),
      array('data' => t('Count'), 'field' => 'hcount'),
      t('Banned IP'),
    );

    // Hostnames(IP) list from watchdog table using the rule.
    $hostnames = autoban_get_hostnames($rule, $header);
    if (!empty($hostnames)) {
      $ind = 0;
      foreach ($hostnames as $hostname) {
        $ip_warn = '';
        $banned_ip = autoban_make_ip_style($hostname->hostname, $rule->ip_type);
        if (_autoban_is_own_ip($banned_ip)) {
          $ip_warn = ' !!!';
          drupal_set_message(t('You may not block your own IP address: @ip.', array('@ip' => ip_address())), 'warning', FALSE);
        }
        $rows[] = array(
          ++$ind,
          $hostname->hostname,
          $hostname->hcount,
          $banned_ip . $ip_warn,
        );
      }

      $build['autoban_ban_form'] = drupal_get_form('autoban_ban_form', $rid);
    }

    $build['autoban_table'] = array(
      '#theme' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => t('Table has no rows!'),
    );
  }
  else {
    drupal_set_message(t('Empty rules list for id=@id.', array('@id' => $rid)), 'error');
  }
  return $build;
}

/**
 * Defines the form for autoban rules.
 *
 * @param int $rid
 *   (optional) Autoban rule id.
 *
 * @ingroup forms
 * @see autoban_form_submit()
 */
function autoban_form($form, $form_state, $rid = NULL) {
  if ($rid == NULL && AUTOBAN_IS_BASE_URL) {
    $form['autoban'] = array(
      '#type' => 'fieldset',
      '#title' => t('Add autoban rule'),
      '#collapsible' => TRUE,
      '#collapsed' => TRUE,
    );
  };

  $form['autoban']['type'] = array(
    '#title' => t('Type'),
    '#type' => 'textfield',
    '#size' => 32,
    '#required' => FALSE,
    '#maxlength' => 64,
    '#description' => t('Type of <a href="/admin/reports/dblog">log message</a>, for example "access denied" or "page not found."'),
  );
  $form['autoban']['message'] = array(
    '#title' => t('Message'),
    '#type' => 'textfield',
    '#size' => 64,
    '#required' => FALSE,
    '#maxlength' => 255,
    '#description' => t('<a href="/admin/reports/dblog">Log message</a> pattern.'),
  );
  $form['autoban']['threshold'] = array(
    '#title' => t('Threshold'),
    '#type' => 'select',
    '#multiple' => FALSE,
    '#options' => drupal_map_assoc(array(
      1,
      2,
      3,
      5,
      10,
      20,
      50,
      100,
      200,
      500,
      1000,
    )
    ),
    '#required' => TRUE,
    '#default_value' => 1,
    '#description' => t('The threshold number of the <a href="/admin/reports/dblog">log entries</a>.'),
  );
  $form['autoban']['user_type'] = array(
    '#title' => t('User type'),
    '#type' => 'select',
    '#multiple' => FALSE,
    '#options' => _autoban_get_user_type_name(),
    '#default_value' => 0,
    '#required' => TRUE,
    '#description' => t('Choose user type.'),
  );
  $form['autoban']['ip_type'] = array(
    '#title' => t('IP type'),
    '#type' => 'select',
    '#multiple' => FALSE,
    '#options' => _autoban_get_ip_type_names_list(),
    '#default_value' => 0,
    '#required' => TRUE,
    '#description' => t('Choose IP type.'),
  );

  $form['rule_id'] = array(
    '#type' => 'value',
    '#value' => $rid,
  );

  $form['autoban']['actions'] = array('#type' => 'actions');

  if ($rid != NULL) {
    // Edit rule.
    $rule = autoban_get_rules($rid);
    $form['autoban']['type']['#default_value']     = filter_xss($rule->type);
    $form['autoban']['message']['#default_value']  = filter_xss($rule->message);
    $form['autoban']['threshold']['#default_value'] = $rule->threshold;
    $form['autoban']['user_type']['#default_value'] = $rule->user_type;
    $form['autoban']['ip_type']['#default_value']  = $rule->ip_type;

    $form['autoban']['actions']['save'] = array(
      '#type' => 'submit',
      '#value' => t('Edit'),
    );
    $form['autoban']['actions']['cancel'] = array(
      '#type' => 'link',
      '#title' => t('Cancel'),
      '#href' => AUTOBAN_BASE_URL,
    );
  }
  else {
    // Add rule.
    $form['autoban']['actions']['submit'] = array(
      '#type' => 'submit',
      '#value' => t('Add'),
    );

    if (!AUTOBAN_IS_BASE_URL) {
      $form['autoban']['actions']['cancel'] = array(
        '#type' => 'link',
        '#title' => t('Cancel'),
        '#href' => AUTOBAN_BASE_URL,
      );
    }
  }

  return $form;
}

/**
 * Form validation for autoban_form().
 */
function autoban_form_validate($form, &$form_state) {
  $type = filter_xss(trim($form_state['values']['type']));
  $message = filter_xss(trim($form_state['values']['message']));
  if (empty($type) && empty($message)) {
    form_set_error('type', t('Type and Message can not both be empty.'));
    form_set_error('message');
  }
}

/**
 * Form submission for autoban_form().
 */
function autoban_form_submit($form, &$form_state) {
  $type = filter_xss(trim($form_state['values']['type']));
  $message = filter_xss(trim($form_state['values']['message']));
  $threshold = $form_state['values']['threshold'];
  $user_type = $form_state['values']['user_type'];
  $ip_type = $form_state['values']['ip_type'];
  $rid = $form_state['values']['rule_id'];

  $fields = array(
    'type' => $type,
    'message' => $message,
    'threshold' => $threshold,
    'user_type' => $user_type,
    'ip_type' => $ip_type,
  );

  if ($rid) {
    // Update record.
    db_update('autoban')
      ->fields($fields)
      ->condition('rid', $rid)
      ->execute();
    $message = t('Autoban rule @rid was updated.', array('@rid' => $rid));
  }
  else {
    // Insert record.
    db_insert('autoban')
      ->fields($fields)
      ->execute();
    $message = t('New autoban rule was inserted.');
  }
  drupal_set_message($message);

  // Redirect to autoban base url.
  if (!AUTOBAN_IS_BASE_URL) {
    $form_state['redirect'] = AUTOBAN_BASE_URL;
  }

  // Reset stored rules.
  drupal_static_reset('autoban_get_rules');
}

/**
 * Autoban rule deletion confirm page.
 *
 * @see autoban_delete_submit()
 */
function autoban_delete($form, &$form_state, $rid) {
  $form['rule_id'] = array(
    '#type' => 'value',
    '#value' => $rid,
  );
  return confirm_form($form,
           t('Are you sure you want to delete Autoban rules %id?', array('%id' => $rid)),
           AUTOBAN_BASE_URL,
           t('This action cannot be undone.'),
           t('Delete'),
           t('Cancel')
         );
}

/**
 * Processes autoban_delete form submissions.
 */
function autoban_delete_submit($form, &$form_state) {
  $rid = $form_state['values']['rule_id'];
  if (!empty($rid)) {
    db_delete('autoban')
      ->condition('rid', $rid)
      ->execute();

    watchdog('autoban', 'Deleted %id from autoban rules.', array('%id' => $rid));
    drupal_set_message(t('The autoban rules %id was deleted.', array('%id' => $rid)));

    // Reset stored rules.
    drupal_static_reset('autoban_get_rules');

    $form_state['redirect'] = AUTOBAN_BASE_URL;
  }
}

/**
 * Ban IP's for all autoban rule.
 *
 * @see autoban_ban_all_form_submit()
 */
function autoban_ban_all_form($form, &$form_state) {
  $form['ban_all'] = array(
    '#type' => 'fieldset',
    '#title' => t('Ban IP for all rules'),
    '#collapsible' => TRUE,
    '#collapsed' => TRUE,
  );
  $form['ban_all']['actions'] = array('#type' => 'actions');
  $form['ban_all']['actions']['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Ban IPs'),
    '#suffix' => t('This action will automatically by cron. Your own IP @ip cannot be blocked.', array('@ip' => ip_address())),
  );
  return $form;
}

/**
 * Form submission for autoban_ban_all_form().
 */
function autoban_ban_all_form_submit($form, &$form_state) {
  // Stored number banned IP to session.
  $_SESSION['autoban_ban_all_count'] = 0;

  $rules = autoban_get_rules();
  if (!empty($rules)) {
    $operations = array();
    foreach ($rules as $rule) {
      $operations[] = array('autoban_ban', array($rule));
    }

    $batch = array(
      'operations' => $operations,
      'finished' => 'autoban_ban_all_batch_finished',
      'title' => t('IP ban'),
      'file' => drupal_get_path('module', 'autoban') . '/autoban.admin.inc',
    );
    batch_set($batch);
  }
}

/**
 * Callback batch finished for ban all batch.
 *
 * @param bool $success
 *   A boolean indicating whether the batch has completed successfully.
 * @param array $results
 *   The value set in $context['results'] by callback_batch_operation().
 * @param array $operations
 *   If $success is FALSE, contains the operations that remained unprocessed.
 */
function autoban_ban_all_batch_finished($success, $results, $operations) {
  if ($success) {
    drupal_set_message(t('Successful IP ban.'));
    if (isset($_SESSION['autoban_ban_all_count'])) {
      drupal_set_message(t('IP ban for all rules has finished. Total IP insertion: @count.',
        array('@count' => $_SESSION['autoban_ban_all_count'])));
    }
  }
  else {
    drupal_set_message(t('Fail IP ban'), 'error');
  }
}

/**
 * Ban IP's for one autoban rule.
 *
 * @see autoban_ban_form_submit()
 */
function autoban_ban_form($form, &$form_state, $rid) {
  $form = array();
  $form['rule_id'] = array(
    '#type' => 'value',
    '#value' => $rid,
  );
  $form['actions'] = array('#type' => 'actions');
  $form['actions']['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Ban IP'),
    '#suffix' => t('Ban IP from the table by inserting to banned IP tables.'),
  );
  return $form;
}

/**
 * Form submission for autoban_ban_form().
 */
function autoban_ban_form_submit($form, &$form_state) {
  $rid = $form_state['values']['rule_id'];
  if (!empty($rid)) {
    $rule = autoban_get_rules($rid);
    $count = autoban_ban($rule);
    drupal_set_message(t('IP ban for the rule has finished. Total IP insertion: @count.',
      array('@count' => $count)));
  }
}
