<?php

namespace Drupal\autocomplete_4xx\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Component\Utility\Xss;
use Drupal\Core\Path\PathValidatorInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Defines a route controller for watches autocomplete form elements.
 */
class AutocompleteSourceController extends ControllerBase {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The node storage.
   *
   * @var \Drupal\node\NodeStorage
   */
  protected $nodeStorage;

  /**
   * The path validator.
   *
   * @var \Drupal\Core\Path\PathValidatorInterface
   */
  protected $pathValidator;

  /**
   * The route provider.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected $routeProvider;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * {@inheritdoc}
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, PathValidatorInterface $path_validator, RouteProviderInterface $route_provider, AccountInterface $current_user) {
    $this->configFactory = $config_factory;
    $this->nodeStroage = $entity_type_manager->getStorage('node');
    $this->pathValidator = $path_validator;
    $this->routeProvider = $route_provider;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    // Instantiates this form class.
    return new static(
      $container->get('config.factory'),
      $container->get('entity_type.manager'),
      $container->get('path.validator'),
      $container->get('router.route_provider'),
      $container->get('current_user')
    );
  }

  /**
   * Handler for autocomplete request.
   */
  public function handleAutocomplete(Request $request) {
    $config = $this->config('autocomplete_4xx.settings');
    $results = [];
    $input = $request->query->get('q');

    // Get the typed string from the URL, if it exists.
    if (!$input) {
      return new JsonResponse($results);
    }

    $input = Xss::filter($input);

    $query = $this->nodeStroage->getQuery()
      ->groupBy('nid')
      ->sort('created', 'DESC')
      ->condition('title', $input, 'CONTAINS');

    // Extend conditions if content types are selected.
    $list = [];
    if (!empty($config->get('content_types'))) {
      foreach ($config->get('content_types') as $content_type) {
        if (!empty($content_type)) {
          $list[] = $content_type;
        }
      }
      if (!empty($list)) {
        $query->condition('type', $list, 'IN');
      }
    }

    $ids = $query->execute();
    $nodes = $ids ? $this->nodeStroage->loadMultiple($ids) : [];
    foreach ($nodes as $node) {
      if ($config->get('include_unpublished')) {
        $results[] = [
          'value' => '/node/' . $node->id(),
          'label' => $node->getTitle() . ' <small>(' . $node->id() . ')</small>',
        ];
      }
      else {
        if ($node->isPublished()) {
          $results[] = [
            'value' => '/node/' . $node->id(),
            'label' => $node->getTitle() . ' <small>(' . $node->id() . ')</small>',
          ];
        }
      }
    }

    if ($config->get('include_routes')) {
      $route_name = $this->routeProvider->getAllRoutes();
      foreach ($route_name as $name => $route) {
        if (strpos($route->getPath(), $input)) {
          if ($config->get('include_parameterized')) {
            $results[] = [
              'value' => $route->getPath(),
              'label' => $route->getPath(),
            ];
          }
          else {
            if (!strpos($route->getPath(), '{') && !strpos($route->getPath(), '}')) {
              $results[] = [
                'value' => $route->getPath(),
                'label' => $route->getPath(),
              ];
            }
          }
        }
      }
    }

    return new JsonResponse($results);
  }

}
