<?php

declare(strict_types = 1);

namespace Drupal\autofill_fields\Hook;

use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountInterface;
use Drupal\user\UserDataInterface;

/**
 * Hook implementations for widgets.
 */
class FormHooks {

  public function __construct(
    private readonly AccountInterface $user,
    private readonly UserDataInterface $userData,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * Implements field_widget_single_element_form_alter().
   */
  #[Hook('field_widget_single_element_form_alter')]
  public function widgetAlter(array &$element, FormStateInterface $form_state, array $context): void {
    // Only run for logged-in users.
    if ($this->user->isAnonymous()) {
      return;
    }
    $entity = $form_state->getFormObject()->getEntity();
    // Only run for new entities; we don't want to overwrite existing data.
    if (!$entity->isNew()) {
      return;
    }
    $plugin = $context['widget'];

    // Check that it is enabled for this field.
    if (!$plugin->getThirdPartySetting('autofill_fields', 'autofill')) {
      return;
    }
    /** @var \Drupal\Core\Field\FieldDefinitionInterface $field_definition */
    $field_definition = $context['items']->getFieldDefinition();
    $type = $field_definition->getType();
    $plugin_id = $plugin->getPluginId();

    $field_id = $entity->getEntityTypeId() . '.' . $entity->bundle() . '.' . $field_definition->getName();
    $value = $this->userData->get('autofill_fields', $this->user->id(), $field_id);
    if (empty($value)) {
      return;
    }
    switch ($type) {
      case 'text_with_summary':
        $element['#default_value'] = $value[0]['value'];
        $element['#format'] = $value[0]['format'];
        $element['summary']['#default_value'] = $value[0]['summary'];
        break;

      case 'entity_reference':
        $value = $value['target_id'] ?? $value;
        switch ($plugin_id) {
          case 'options_select':
          case 'options_buttons':
          case 'select2_entity_reference':
            $element['#default_value'] = [];
            foreach ($value as $reference) {
              $element['#default_value'][] = $reference['target_id'];
            }
            break;

          case 'entity_reference_autocomplete':
            if (!$value[$element['target_id']['#delta']]['target_id']) {
              return;
            }
            $element['target_id']['#default_value'] = $this->entityTypeManager
              ->getStorage($element['target_id']['#target_type'])
              ->load($value[$element['target_id']['#delta']]['target_id']);
            break;

          case 'entity_reference_autocomplete_tags':
            $element['target_id']['#default_value'] = [];
            foreach ($value as $reference) {
              $element['target_id']['#default_value'][] = $this->entityTypeManager
                ->getStorage($element['target_id']['#target_type'])
                ->load($reference['target_id']);
            }
            break;
        }
        break;

      case 'decimal':
        if ($plugin_id === 'number') {
          $element['value']['#default_value'] = $value[$element['value']['#delta']]['value'];
        }
        break;

      case 'datetime':
        if ($plugin_id === 'datetime_default') {
          $element['value']['#default_value'] = new DrupalDateTime($value);
        }
        break;

      case 'list_string':
        if ($plugin_id === 'options_select') {
          $element['#default_value'] = array_column($value, 'value');
        }
        break;

      default:
        $element['#default_value'] = $value;
    }
  }

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter')]
  public function formAlter(array &$form, FormStateInterface $form_state, string $form_id): void {
    $form_object = $form_state->getFormObject();
    if (method_exists($form_object, 'getEntity')) {
      $entity = $form_object->getEntity();
      if (is_a($entity, ContentEntityInterface::class)) {
        $form['#submit'][] = [$this, 'submit'];
        if (isset($form['actions']['publish'])) {
          $form['actions']['publish']['#submit'][] = [$this, 'submit'];
        }
        else {
          $form['actions']['submit']['#submit'][] = [$this, 'submit'];
        }
      }
    }
  }

  /**
   * Submit function to save field values per entity create/edit action.
   */
  public function submit(array $form, FormStateInterface $form_state): void {
    if (!$user_id = \Drupal::currentUser()->id()) {
      return;
    }

    // Ensure that the form entity is properly loaded.
    /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
    if (!$entity = $form_state->getFormObject()->getEntity()) {
      return;
    }

    // @todo get real form mode.
    $form_mode = 'default';
    $entity_form_display = EntityFormDisplay::load($entity->getEntityTypeId() . '.' . $entity->bundle() . '.' . $form_mode);
    foreach ($entity_form_display->getComponents() as $name => $component) {
      if (!isset($component['third_party_settings']['autofill_fields']['autofill'])) {
        continue;
      }
      if ($component['type'] === 'datetime_default') {
        $value = $form_state->getValue($name)[0]['value']->format('c');
      }
      else {
        $value = $form_state->getValue($name);
      }
      $this->userData->set('autofill_fields', $this->user->id(), $entity->getEntityTypeId() . '.' . $entity->bundle() . '.' . $name, $value);
      $component['third_party_settings']['autofill_fields']['stored'][$user_id] = $value;
    }

  }

}
