<?php

namespace Drupal\autogrid\Form;

use Drupal\autogrid\EntityHelper;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Class SettingsForm.
 *
 * @package Drupal\autogrid\Form
 */
class EntitiesForm extends FormBase implements ContainerInjectionInterface {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * @var \Drupal\autogrid\EntityHelper
   */
  protected $entityHelper;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('autogrid.entity_helper'),
    );
  }

  /**
   * Constructs a new CloudFlareAdminSettingsForm.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\autogrid\EntityHelper $entity_helper
   *   Tracks rate limits associated with CloudFlare API.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityHelper $entity_helper) {
    $this->configFactory = $config_factory;
    $this->entityHelper = $entity_helper;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'autogrid.settings',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'autogrid_admin';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->configFactory->get('autogrid.settings');
    $entity_type_labels = [];
    foreach ($this->entityHelper->getSupportedEntityTypes() as $entity_type_id => $entity_type) {
      $entity_type_labels[$entity_type_id] = $entity_type->getLabel() ? : $entity_type_id;
    }
    asort($entity_type_labels);

    $form['entity_types'] = [
      '#type' => 'checkboxes',
      '#description' => $this->t('Autogrid will be generated only for entity types enabled here. For all entity types featuring bundles (e.g. <em>node</em>) Autogrid settings have to be set on their bundle pages (e.g. <em>page</em>).'),
      '#options' => $entity_type_labels,
      '#default_value' => $config->get('entity_types'),
    ];

    $form['save'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->configFactory->getEditable('autogrid.settings');
    $config
      ->set('entity_types', $form_state->getValue('entity_types'));
    $config->save();

    $this->messenger()->addMessage($this->t('Configuration was saved. Please clear the cache to see the new links to manage data.'));
  }

}
