<?php

namespace Drupal\autogrid\Routing;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Drupal\Core\Routing\RouteSubscriberBase;
use Drupal\Core\Routing\RoutingEvents;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

/**
 * Subscriber for auto_entitylabel routes.
 */
class RouteSubscriber extends RouteSubscriberBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The router service.
   *
   * @var \Symfony\Component\Routing\RouterInterface
   */
  protected $routeProvider;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new RouteSubscriber object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Routing\RouteProviderInterface $router_provider
   *   The router provider service.
   */
  public function __construct(ConfigFactoryInterface $config_factory, EntityTypeManagerInterface $entity_type_manager, RouteProviderInterface $router_provider) {
    $this->configFactory = $config_factory;
    $this->entityTypeManager = $entity_type_manager;
    $this->routeProvider = $router_provider;
  }

  /**
   * {@inheritdoc}
   */
  protected function alterRoutes(RouteCollection $collection) {
    $config = $this->configFactory->get('autogrid.settings')?->get('entity_types');
    if (!$config) {
      $config = [];
    }
    foreach ($config as $entity_type_id => $value) {
      if ($value == 0) {
        continue;
      }
      $entity_type = $this->entityTypeManager->getDefinition($entity_type_id);
      if (!$entity_type) {
        continue;
      }
      if (method_exists($entity_type, 'getBundleEntityType') && $bundle_type = $entity_type->getBundleEntityType()) {
        // If the entity type uses bundles, use the bundle type instead.
        $entity_type = $this->entityTypeManager->getDefinition($bundle_type);
        $entity_type_id = $bundle_type;
      }
      if ($route = $this->getEntityGridRoute($entity_type)) {
        $collection->add("entity.$entity_type_id.autogrid", $route);
      }
    }
  }

  /**
   * Gets the Autogrid route.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type.
   *
   * @return \Symfony\Component\Routing\Route|null
   *   The generated route, if available.
   */
  protected function getEntityGridRoute(EntityTypeInterface $entity_type) {
    // Validate that an existing route is available as a template.
    $link_template = $entity_type->getLinkTemplate('edit-form') ? 'edit-form' : 'canonical';
    if (!$route_load = $entity_type->getLinkTemplate($link_template)) {
      return NULL;
    }

    // Check if the entity type uses bundles.
    if (method_exists($entity_type, 'getBundleOf') && $parent_type = $entity_type->getBundleOf()) {
      // If bundles used, set appropriate variables.
      $entity_type_id = $parent_type;
      $bundle_type = $entity_type->id();
      $table_method = 'getBundleTable';
      $title_method = 'getBundleTitle';
    }
    else {
      // If no bundles used, use the same entity type for both.
      $entity_type_id = $entity_type->id();
      $bundle_type = $entity_type_id;
      $table_method = 'getTable';
      $title_method = 'getTitle';
    }
    $route = new Route($route_load . '/grid');
    $route
      ->addDefaults([
        '_controller' => '\Drupal\autogrid\Controller\GridController::' . $table_method,
        '_title_callback' => '\Drupal\autogrid\Controller\GridController::' . $title_method,
        'entity_type_id' => $entity_type_id,
      ])
      ->addRequirements([
        '_permission' => 'view autogrid',
      ])
      ->setOption('_field_ui', TRUE)
      ->setOption('parameters', [
        $bundle_type => ['type' => 'entity:' . $bundle_type],
      ]);
    return $route;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = parent::getSubscribedEvents();
    $events[RoutingEvents::ALTER] = ['onAlterRoutes', -90];

    return $events;
  }

}
