<?php

/**
 * @file
 * Defines text processing classes for the Autolink module.
 */

/**
 * Defines the Autolink node processor interface.
 */
interface AutolinkProcessorInterface {
  function save();
  function update();
  function clear($limit = AUTOLINK_DEFAULT_LIMIT, $link = NULL);
}

/**
 * Defines the text processing class for Autolink module.
 *
 * The Autolink processor class is abstract and thuse requires a
 * child processor class to operate. Additionally, a plugin should
 * be set with the setPlugin() method.
 *
 * @ingroup autolink
 */
abstract class AutolinkProcessor implements AutolinkProcessorInterface {
  protected static $links = array();
  protected $collector;

  /**
   * Adds plugin and related link type information to the object.
   */
  function __construct() {}

  /**
   * Resets the properties of the object.
   */
  function __destruct() {
    $this->info = NULL;
    $this->plugin = NULL;
    $this->collector = NULL;
  }

  /**
   * Magic caller method for calling plugin methods.
   */
  function __call($method, $args) {
    if (method_exists($this->plugin, $method)) {
      return call_user_func_array(array($this->plugin, $method), $args);
    }
  }

  /**
   * Sets the current plugin object.
   */
  function setPlugin($plugin) {
    $this->plugin = $plugin;
  }

  /**
   * Updates links on a node by removing old links and generating new ones.
   */
  function update() {
    $handler = new AutolinkErrorHandler();
    try {
      $this->process($this->getLinks());
    }
    catch (AutolinkException $e) {
      $e->logMessage($handler);
    }
    $handler->displayAll();
    return $this;
  }

  /**
   * The process method passes text to the text processor to generate links.
   */
  abstract protected function process($links);

  /**
   * Adds links to a node field.
   */
  protected function &processText(&$text, $links, &$total, $total_limit = 'all', $link_limit = 'all') {
    foreach ($links as $lid => $link) {
      // Break the loop if the total limit has already been reached.
      if ($total_limit != 'all' && $total >= $total_limit) {
        break;
      }
      else {
        $text = preg_replace($this->fetchExpression($link), '$this->buildReplacement($link, "$1")', $text, $this->getLimit($total_limit, $link_limit, $total), $count);
        $total = $total + $count;
      }
    }
  }

  /**
   * Returns links for the current node.
   */
  protected function getLinks() {
    if (!isset(self::$links[$this->node->type])) {
      Autolink::loadFiles(array('links'));
      $this->collector = new AutolinkLinkCollector($this->node->type);
      self::$links[$this->node->type] = $this->collector->getLinks();
    }
    return self::$links[$this->node->type];
  }

  /**
   * Returns a limit based on plugin and link limits.
   */
  private function getLimit($total_limit, $link_limit, $total) {
    if ($total_limit == 'all' && $link_limit == 'all') {
      $limit = -1;
    }
    elseif ($total_limit == 'all' && $link_limit != 'all') {
      $limit = $link_limit;
    }
    elseif ($total_limit != 'all' && $link_limit == 'all') {
      $limit = $total_limit;
    }
    if ($total_limit != 'all' && $link_limit != 'all') {
      if ($link_limit > ($total_limit - $total)) {
        $limit = $total_limit - $total;
      }
      else {
        $limit = $link_limit;
      }
    }
    return $limit;
  }

  /**
   * Clears a string of Autolink links.
   */
  protected function &clearText(&$text, $limit = AUTOLINK_DEFAULT_LIMIT, $link = NULL) {
    // Record the current node field for comparison.
    $previous = $text;
    // Loose check to see if the 'autolink' class might exist in the text.
    if (strstr($text, 'autolink')) {
      $dom = new DOMDocument();
      if ($dom->loadHTML($text)) {
        $anchors = $dom->getElementsByTagName('a');
        $length = $anchors->length;
        // Loop through each existing anchor to see if it needs to be removed.
        if ($length > 0) {
          $i = $length - 1;
          while ($i > -1) {
            $anchor = $anchors->item($i);
            if ($limit == AUTOLINK_DEFAULT_LIMIT) {
              // Only remove anchors that have the 'autolink' class.
              if ($anchor->hasAttribute('class') && $anchor->getAttribute('class') == 'autolink' || strstr($anchor->getAttribute('class'), 'autolink')) {
                $text = $anchor->nodeValue;
                $textNode = $dom->createTextNode($text);
                $anchor->parentNode->replaceChild($textNode, $anchor);
              }
            }
            elseif ($limit == 'limit' && $anchor->nodeValue == drupal_strtolower($link->getText(drupal_strtolower($link->getKeyword)))) {
              // Only remove anchor tags that match the given link.
              if ($anchor->hasAttribute('class') && $anchor->getAttribute('class') == 'autolink' || strstr($anchor->getAttribute('class'), 'autolink')) {
                $text = $anchor->nodeValue;
                if ($anchor->getAttribute('href') == $link->getPath() && $text == $link->getKeyword()) {
                  $textNode = $dom->createTextNode($text);
                  $anchor->parentNode->replaceChild($textNode, $anchor);
                }
              }
            }
            $i--;
          }
        }
        // Save the text and return TRUE if it has changed.
        $body = $dom->GetElementsbyTagName('body')->item(0);
        // Following portion from the patch at issue #1170430.
        if ($body instanceof DOMNode) {
          // PHP 5.2 compatibility, saveHTML does not accept args.
          $return = new DOMDocument;
          $root = $return->createElement('body');
          $root = $return->appendChild($root);
          $result_node = $return->importNode($body, TRUE);
          $return->documentElement->appendChild($result_node);
          $out = $return->saveHTML();
          $out = $this->replaceTags($out);
          $text = $out;
        }
        unset($dom);
        if ($text != $previous) {
          return TRUE;
        }
      }
    }
  }

  /**
   * Replaces unwanted HTML tags after links have been removed.
   * This is done to remove tags that were placed by the HTML DOM node.
   */
  private function replaceTags($text) {
    // Replace all body tags.
    $text = str_replace(array('<body>', '</body>'), '', $text);

    // Replace paragraph <p></p> tags if they appear at the beginning or end
    // of the body of text.
    foreach (array('<p>', '</p>') as $string) {
      if (drupal_substr($text, drupal_strlen($text) - drupal_strlen($string) - 1, drupal_strlen($string)) == $string) {
        $text = substr_replace($text, '', drupal_strlen($text) - drupal_strlen($string) - 1, drupal_strlen($string));
      }
      elseif (drupal_substr($text, 0, drupal_strlen($string)) == $string) {
        $text = substr_replace($text, '', 0, drupal_strlen($string));
      }
    }
    return $text;
  }
}

/**
 * Defines the Autolink node processor class.
 *
 * The node processor loads a node, which is passed by reference
 * because it is an object. The process() method calls a getFields()
 * method, so plugins that use the node processor must implement
 * a getFields() method to identify which fields to process. It
 * also provides a clear() method for removing links.
 *
 * @ingroup autolink
 */
class AutolinkNodeProcessor extends AutolinkProcessor {
  public $node = NULL;

  /**
   * Loads a node into the processor object for processing.
   */
  function load($node) {
    $this->node = $node;
    $this->plugin->setItem($node);
    return $this;
  }

  /**
   * Returns a processed node for saving.
   */
  function save() {
    node_save($this->node);
    return $this->node;
  }

  /**
   * Adds links to a node.
   *
   * Fields are retrieved from the plugin via the getFields() method.
   * Plugins that make use of this processor should always implement that
   * method. Also returns a value indicating whether the text was updated
   * so $processor->save() can be conditionally called.
   */
  protected function process($links) {
    $total = 0;
    $updated = FALSE;
    $link_limit = $this->getLinkLimit();
    $total_limit = $this->getTotalLimit();
    foreach ($this->getFields($this->node->type) as $field) {
      if ($field != 'body' && $field != 'teaser' && isset($this->node->content[$field])) {
        $this->clearText($this->node->content[$field]);
        $this->processText($this->node->content[$field], $links, $total, $total_limit, $link_limit);
      }
      elseif (isset($this->node->{$field})) {
        $this->clearText($this->node->{$field});
        $this->processText($this->node->{$field}, $links, $total, $total_limit, $link_limit);
      }
      else {
        return FALSE;
      }
    }
    return $updated;
  }

  /**
   * Clears a node of all Autolink links.
   *
   * @return
   *   TRUE if the node has changed. FALSE if not.
   */
  function clear($limit = AUTOLINK_DEFAULT_LIMIT, $link = NULL) {
    $updated = FALSE;
    foreach ($this->getFields($this->node->type) as $field) {
      switch ($limit) {
        case 'all':
          $this->clearText($this->node->{$field});
          break;
        case 'limit':
          $this->clearText($this->node->{$field}, 'limit', $link);
          break;
      }
    }
    return $updated;
  }
}

/**
 * Defines the Autolink text processor class, used for filters.
 *
 * The text processor loads a string that is passed by reference,
 * so no return is necessary. It also accepts a related node to
 * use for gathering information on the text.
 *
 * @ingroup autolink
 */
class AutolinkTextProcessor extends AutolinkProcessor {
  public $text = NULL;
  public $node = NULL;

  /**
   * Loads a node into the processor object for processing.
   */
  function load(&$text, $node) {
    $this->text = &$text;
    $this->node = $node;
    $this->plugin->setItem($node);
    return $this;
  }

  /**
   * Returns a processed node for saving.
   */
  function save() {
    return $this->text;
  }

  /**
   * Adds links to a node.
   */
  protected function process($links) {
    $total = 0;
    $updated = FALSE;
    $previous = $this->text;
    $this->processText($this->text, $links, $total);
    // Indicate whether the node has been changed.
    if ($text != $previous) {
      $updated = TRUE;
    }
    return $updated;
  }

  /**
   * Clears text of all Autolink links.
   *
   * @return
   *   TRUE if the node has changed. FALSE if not.
   */
  function clear($limit = AUTOLINK_DEFAULT_LIMIT, $link = NULL) {
    $this->clearText($this->text, $limit, $link);
  }

}

/**
 * Defines the Autolink plugin interface.
 */
interface AutolinkPluginInterface {
  function getLinkTypes();
  function getTotalLimit();
  function getLinkLimit();
  function getExpression(AutolinkLink $link);
  function getCase(AutolinkLink $link);
  function getAttributes(AutolinkLink $link);
  function getText(AutolinkLink $link, $match);
  function buildReplacement(AutolinkLink $link, $match);
}

/**
 * Defines the base class for Autolink plugins. This class is only
 * abstract so that it must be extended.
 *
 * @ingroup autolink
 */
abstract class AutolinkPlugin implements AutolinkPluginInterface {
  public $_info = array();
  // Items can hold full nodes or individual pieces of text.
  public $item;
  protected $config;
  public $fields = array();
  const EXPRESSION_BEFORE = '/(?!(?:[^<]+>|[^>]+<\/a>))';
  const EXPRESSION_SENSITIVE = '/e';
  const EXPRESSION_INSENSITIVE = '/ie';

  /**
   * Standard constructor method.
   */
  function __construct($name) {
    $this->config = autolink_get('config');
    $this->_info = autolink_get('PluginInfo')->getCurrent();
  }

  /**
   * Destructor method.
   */
  function __destruct() {
    $this->item = NULL;
    $this->_info = array();
    $this->config = NULL;
  }

  /**
   * Magic call method to call configuration methods if method can't be found.
   */
  function __call($method, $args) {
    if (method_exists($this->config, $method)) {
      return call_user_func_array(array($this->config, $method), $args);
    }
  }

  /**
   * Sets the item property which holds the item being processed.
   */
  function setItem($item) {
    $this->item = $item;
  }

  /**
   * Returns an array of link types supported by the plugin.
   */
  function getLinkTypes() {
    return $this->_info['link types'];
  }

  /**
   * Retrieves the total limit per node as defined by plugin. If setTotalLimit()
   * method is not defined, there will be no link limit.
   */
  function getTotalLimit() {
    return AUTOLINK_DEFAULT_LIMIT;
  }

  /**
   * Retrieves the per link limit defined by the plugin. If no setLinkLimit()
   * method is defined, there will be no limit placed on links.
   */
  function getLinkLimit() {
    return AUTOLINK_DEFAULT_LIMIT;
  }

  /**
   * Retrieves a regular expression defined by the plugin.
   *
   * The regular expression provided is modified by prepending and appending
   * default regular expression characters that ensure links are not generated
   * on top of other links and full words are required.
   */
  function fetchExpression(AutolinkLink $link) {
    $before = self::EXPRESSION_BEFORE;
    $after = $this->getCase($link) ? self::EXPRESSION_SENSITIVE : self::EXPRESSION_INSENSITIVE;
    $additions = $this->getExpression($link);
    return $before . $additions['prepend'] .'('. preg_quote($link->get('keyword')) .')'. $additions['append'] . $after;
  }

  /**
   * Defines the default regular expression array. Values are empty strings.
   */
  function getExpression(AutolinkLink $link) {
    return array('prepend' => '', 'append' => '');
  }

  /**
   * Defines the default case-sensitivity setting. Non-case-sensitive.
   */
  function getCase(AutolinkLink $link) {
    return FALSE;
  }

  /**
   * Defines the default attributes array.
   */
  function getAttributes(AutolinkLink $link) {
    return array('title' => $this->get('keyword'));
  }

  /**
   * Returns the text of a link based on the matched text.
   */
  function getText(AutolinkLink $link, $match) {
    return $match;
  }

  /**
   * Build a link by retrieving attributes from the link object.
   */
  function buildReplacement(AutolinkLink $link, $match) {
    $attributes = $this->getAttributes($link);

    // All links are tagged with the class 'autolink' for reference by the module.
    $attributes['class'] = !empty($attributes['class']) ? 'autolink '. $attributes['class'] : 'autolink';
    $options = array(
      'query' => $link->get('query'),
      'absolute' => TRUE,
      'alias' => FALSE,
      'attributes' => isset($attributes) && !empty($attributes) ? $attributes : '',
    );

    // Unset the empty attributes to avoid invalid markup and unnecessary attributes.
    if (!empty($options['attributes'])) {
      foreach ($options['attributes'] as $attribute => $value) {
        if (empty($value)) {
          unset($options['attributes'][$attribute]);
        }
      }
    }
    else {
      unset($options['attributes']);
    }

    // The target attribute causes _blank if 'none' is not unset.
    if ($options['attributes']['target'] == 'none') {
      unset($options['attributes']['target']);
    }
  
    // The l() function will build the proper url using arguments in options.
    $text = $this->getText($link, $match);
    $anchor = l($text, $link->get('path'), $options);
    return $anchor;
  }
}
