<?php

/**
 * @file
 * Defines automatic linking classes and provider classes for
 * Autolink Automatic Linking module.
 */

/**
 * Defines the automatic node title link class.
 */
class AutolinkAutoNodeLink extends AutolinkLink {
  /**
   * Returns an array of link attributes.
   */
  function getAttributes() {
    $parts = explode('/', $this->get('path'));
    if (module_exists('token')) {
      // This method gave menu.module errors without loading the node FIRST
      // and then checking its existence with a conditional.
      $node = node_load($parts[1]);
      if ($node) {
        $pattern = $this->config->get('node_title', '[title]');
        $title = !empty($pattern) ? token_replace($pattern, 'node', $node) : $this->get('keyword');
      }
    }
    else {
      $title = $this->get('keyword');
    }

    return array(
      'title' => $title,
      'style' => $this->config->get('autolink_style', ''),
      'class' => $this->config->get('autolink_class', ''),
      'target' => $this->config->get('target', 'none'),
    );
  }
}

/**
 * Defines the automatic node title provider class.
 */
class AutolinkAutoNodeProvider extends AutolinkProvider {
  /**
   * Returns an array of generic link information for processing.
   */
  function provideLinks($type) {
    $nodes = array();
    if ($this->config->fetchArraySetting('auto_node_types')->isEnabled($type)) {
      $types = $this->config->fetchArraySetting('auto_node_types_destination')->getEnabled();
      if (!empty($types)) {
        $result = db_query('SELECT * FROM {node} WHERE type IN ('. db_placeholders($types, 'varchar') .')', $types);
        while ($node = db_fetch_object($result)) {
          $nodes[$node->nid] = array('keyword' => $node->title, 'path' => 'node/'. $node->nid);
        }
      }
    }
    return $nodes;
  }
}

/**
 * Defines the automatic taxonomy term link class.
 */
class AutolinkAutoTermLink extends AutolinkLink {
  /**
   * Returns an array of link attributes.
   */
  function getAttributes() {
    $parts = explode('/', $this->get('path'));
    if ($term = $this->termLoad($parts[2]) && module_exists('token')) {
      $pattern =  $this->config->get('term_title', '[term]');
      $title = !empty($pattern) ? token_replace($pattern, 'term', $term) : $this->get('keyword');
    }
    else {
      $title = $this->get('keyword');
    }

    return array(
      'title' => $title,
      'style' => $this->config->get('autolink_style', ''),
      'class' => $this->config->get('autolink_class', ''),
      'target' => $this->config->get('target', 'none'),
    );
  }

  /**
   * Loads a taxonomy term for use in token replacements.
   */
  private function termLoad($tid) {
    return db_fetch_object(db_query('SELECT * FROM {term_data} WHERE tid = %d', $tid));
  }
}

/**
 * Defines the automatic taxonomy term provider class.
 */
class AutolinkAutoTermProvider extends AutolinkProvider {
  /**
   * Returns an array of generic link information for processing.
   */
  function provideLinks($type) {
    $terms = array();
    if ($this->config->fetchArraySetting('auto_node_types')->isEnabled($type)) {
      $vids = $this->config->fetchArraySetting('auto_vocabularies')->getEnabled();
      if (!empty($vids)) {
        $result = db_query('SELECT * FROM {term_data} WHERE vid IN ('. db_placeholders($vids) .')', $vids);
        while ($term = db_fetch_object($result)) {
          $terms[$term->tid] = array('keyword' => $term->name, 'path' => 'taxonomy/term/'. $term->tid);
        }
      }
    }
    return $terms;
  }
}

/**
 * Defines the automatic user name link class.
 */
class AutolinkAutoUserLink extends AutolinkLink {
  /**
   * Returns an array of link attributes.
   */
  function getAttributes() {
    $parts = explode('/', $this->get('path'));
    if ($user = user_load(array('uid' => $parts[1])) && module_exists('token')) {
      $pattern = $this->config->get('user_title', '[user]');
      $title = !empty($pattern) ? token_replace($pattern, 'user', $user) : $this->get('keyword');
    }
    else {
      $title = $this->get('keyword');
    }

    return array(
      'title' => $title,
      'style' => $this->config->get('autolink_style', ''),
      'class' => $this->config->get('autolink_class', ''),
      'target' => $this->config->get('target', 'none'),
    );
  }
}

/**
 * Defines the automatic user name provider class.
 */
class AutolinkAutoUserProvider extends AutolinkProvider {
  /**
   * Returns an array of generic link information for processing.
   */
  function provideLinks($type) {
    $users = array();
    if ($this->config->fetchArraySetting('auto_node_types')->isEnabled($type)) {
      $roles = $this->config->fetchArraySetting('auto_user_roles')->getEnabled();
      if (!empty($roles)) {
        $result = db_query('SELECT * FROM {users} u LEFT JOIN {users_roles} r ON u.uid = r.uid WHERE r.rid IN ('. db_placeholders($roles) .')', $roles);
        while ($user = db_fetch_object($result)) {
          $users[$user->uid] = array('keyword' => $user->name, 'path' => 'user/'. $user->uid);
        }
      }
    }
    return $users;
  }
}
