<?php

/**
 * @file
 * Provides form classes for the Autolink Default Plugin.
 */

/**
 * Defines a class for providing common elements to child classes.
 */
abstract class AutolinkDefaultLinkForm extends AutolinkLinkForm {

  /**
   * Provides a field for defining the keyword for each link.
   */
  protected function fieldKeyword($settings) {
    $form = array();
    $form['keyword'] = array(
      '#type' => 'textfield',
      '#title' => t('Keyword(s)'),
      '#default_value' => isset($settings['keyword']) ? $settings['keyword'] : '',
      '#required' => $keyword_required,
      '#disabled' => $keyword_disabled,
      '#description' => $keyword_disabled == TRUE ? NULL :
        t('The keyword you would like converted into a link. This can be more than one word'),
    );
    return $form;
  }

  /**
   * Provides a field for setting case-sensitivity per link.
   */
  protected function fieldCase($settings) {
    $form = array();
    $form['case'] = array(
      '#type' => 'checkbox',
      '#title' => t('Case-sensitive keyword matching'),
      '#default_value' => isset($settings['case']) ? $settings['case'] : 0,
      '#description' => t('Words will only be matched if they match letter case. i.e. <i>AutoLink</i> ' .
                          'is not the same as <i>autolink</i> if case-sensitivity is enabled.'),
    );
    return $form;
  }

  /**
   * Default methods for setting values.
   */

  /**
   * Returns the keyword.
   */
  function setKeyword($form_values) {
    return $form_values['keyword']; 
  }

  /**
   * Returns the link path.
   */
  function setPath($form_values) {
    return $form_values['path'];
  }

  /**
   * Returns form values to be saved in the database.
   */
  function setExtra($form_values) {
    return array(
      'case' => $form_values['case'],
    );
  }

}

/**
 * Defines the external URL form class.
 */
class AutolinkExternalForm extends AutolinkDefaultLinkForm {

  /**
   * Defines the link settings form.
   */
  function form($settings) {
    $form = $this->fieldKeyword($settings);
    $form['path'] = array(
      '#type' => 'textfield',
      '#title' => t('URL'),
      '#required' => TRUE,
      '#default_value' => isset($settings['path']) ? $settings['path'] : '',
      '#description' => t('e.g. <i>http://www.google.com</i>'),
    );
    $form += $this->fieldCase($settings);
    return $form;
  }

  /**
   * Validates form values.
   */
  function validate($form_values) {
    if (!menu_path_is_external($form_values['path'])) {
      form_set_error('path', t('The url <i>%url</i> is invalid. You must provide a valid external url.', array('%url' => $form_values['path'])));
    }
  }

}

/**
 * Defines the node link form class.
 */
class AutolinkNodeTitleForm extends AutolinkDefaultLinkForm {

  /**
   * Defines the link settings form.
   */
  function form($settings) {
    // Get the node ID from the link_path to set default values.
    if (isset($settings['path']) && !empty($settings['path'])) {
      $args = explode('/', $settings['path']);
      $nid = $args[1];
    }

    $titles = self::getTitles();
    $select['select'] = t('--Select a node--');
    $select += $titles;

    $form = array();
    $form = $this->fieldKeyword($settings);
    $form['title'] = array(
      '#type' => 'select',
      '#title' => t('Select a node title'),
      '#options' => $select,
      '#default_value' => isset($nid) ? $nid : NULL,
    );
    $form += $this->fieldCase($settings);
    return $form;
  }

  /**
   * Validates form values.
   */
  function validate($form_values) {
    if ($values['title'] == 'select') {
      form_set_error('title', t('You must select content'));
    }
  }

  /**
   * Sets the link path.
   */
  function setPath($form_values) {
    return 'node/'. $form_values['title'];
  }

  /**
   * Returns an array of node titles for use in a select list.
   */
  private static function getTitles() {
    $nodes = array();
    $result = db_query('SELECT n.nid, n.title FROM {node} n');
    while ($node = db_fetch_object($result)) {
      $nodes[$node->nid] = $node->title;
    }
    asort($nodes);
    return $nodes;
  }
}

/**
 * Defines the node path form class.
 */
class AutolinkNodePathForm extends AutolinkDefaultLinkForm {

  /**
   * Defines the link settings form.
   */
  function form($settings) {
    $form = array();
    $form = $this->fieldKeyword($settings);
    $path = isset($settings['path']) ? autolink_ui_get_path($settings['path']) : '';
    $form['path'] = array(
      '#type' => 'textfield',
      '#title' => t('Enter a path'),
      '#required' => TRUE,
      '#default_value' => $path,
      '#description' => t('Provide a valid Drupal node path, normal or aliased.<br />' .
                          'Example: \'node/12\' or \'foo/bar\''),
    );
    $form += $this->fieldCase($settings);
    return $form;
  }

  /**
   * Validates form values.
   */
  function validate($form_values) {
    // Validate the path as an internal aliased or normal Drupal path.
    if (!menu_valid_path(array('link_path' => $form_values['path'])) && !drupal_lookup_path('source', $form_values['path']) && $form_values['path'] !== 'contact') {
      form_set_error('path', t('The path <i>%path</i> is invalid. You must provide a valid internal Drupal path.', array('%path' => $form_values['path'])));
    }
  }

}

/**
 * Defines a helper class for user links.
 */
abstract class AutolinkUserLinkForm extends AutolinkDefaultLinkForm {

  /**
   * Returns an array of users for listing in a select list.
   */
  protected static function getUsers() {
    $users = array();
    $result = db_query('SELECT u.uid, u.name FROM {users} u WHERE u.uid <> 0');
    while ($user = db_fetch_object($result)) {
      $users[$user->uid] = $user->name;
    }
    asort($users);
    return $users;
  }

  /**
   * Returns a specific user object.
   */
  protected static function getUser($path) {
    $args = explode('/', $path);
    $uid = $args[1];
    $user = is_numeric($uid) ? user_load(array('uid' => $uid)) : NULL;
    return $user;
  }
}

/**
 * Defines the user select form class.
 */
class AutolinkUserSelectForm extends AutolinkUserLinkForm {

  /**
   * Defines the link settings form.
   */
  function form($settings) {
    $form = $this->fieldKeyword($settings);
    if (isset($settings['path'])) {
      $user = parent::getUser($settings['path']);
    }

    $users = parent::getUsers();

    $form['user'] = array(
      '#type' => 'select',
      '#title' => t('Users'),
      '#required' => TRUE,
      '#options' => $select,
      '#default_value' => isset($uid) ? $uid : 'select',
      '#description' => t('Links will link to the user profile of the selected user.'),
    );
    $form += $this->fieldCase($settings);
    return $form;
  }

  /**
   * Validates form values.
   */
  function validate($form_values) {
    if ($form_values['user'] == 'select') {
      form_set_error('user', t('You must select a user.'));
    }
  }

  /**
   * Returns teh link path.
   */
  function setPath($form_values) {
    $path = 'user/'. $form_values['user'];
    return $path;
  }

}

/**
 * Defines the user name entry form class.
 */
class AutolinkUserNameForm extends AutolinkUserLinkForm {

  /**
   * Defines the link settings form.
   */
  function form($settings) {
    $form = $this->fieldKeyword($settings);
    if (isset($settings['path'])) {
      $user = parent::getUser($settings['path']);
    }

    $form['user_name'] = array(
      '#type' => 'textfield',
      '#title' => t('Enter a username'),
      '#required' => TRUE,
      '#default_value' => isset($user) && !is_null($user) ? $user->name : '',
    );
    $form += $this->fieldCase($settings);
    return $form;
  }

  /**
   * Validates form values.
   */
  function validate($form_values) {
    if (user_load(array('name' => $form_values['user_name'])) == FALSE) {
      form_set_error('user_name', t('The user \'%user\' does not exist.', array('%user' => $form_values['user_name'])));
    }
  }

  /**
   * Returns the link path.
   */
  function setPath($form_values) {
    $user = user_load(array('name' => $form_values['user_name']));
    $path = 'user/'. $user->uid;
    return $path;
  }

}

/**
 * Defines the user mail entry form class.
 */
class AutolinkUserMailForm extends AutolinkUserLinkForm {

  /**
   * Defines the link settings form.
   */
  function form($settings) {
    $form = $this->fieldKeyword($settings);
    if (isset($settings['path'])) {
      $user = parent::getUser($settings['path']);
    }

    $form['user_mail'] = array(
      '#type' => 'textfield',
      '#title' => t('Enter a user e-mail address'),
      '#required' => TRUE,
      '#default_value' => isset($user) && !is_null($user) ? $user->mail : '',
    );
    $form += $this->fieldCase($settings);
    return $form;
  }

  /**
   * Validates form values.
   */
  function validate($form_values) {
    if (user_load(array('mail' => $form_values['user_mail'])) == FALSE) {
      form_set_error('user_mail', t('Cannot locate user data for user email \'%email\'.', array('%email' => $form_values['user_mail'])));
    }
  }

  /**
   * Returns the link path.
   */
  function setPath($form_values) {
    $user = user_load(array('mail' => $form_values['user_mail']));
    $path = 'user/'. $user->uid;
    return $path;
  }

}

/**
 * Defines the taxonomy term selection form class.
 */
class AutolinkTermSelectForm extends AutolinkDefaultLinkForm {

  /**
   * Defines the link settings form.
   */
  function form($settings) {
    $form = $this->fieldKeyword($settings);
    if (isset($settings['path'])) {
      $args = explode('/', $settings['path']);
    }

    $select['select'] = t('--Select a term--');
    $terms = self::getTerms();
    $select += $terms;

    $form['term_select'] = array(
      '#type' => 'select',
      '#title' => t('Select a term'),
      '#required' => TRUE,
      '#options' => $select,
      '#default_value' => isset($args[2]) ? $args[2] : 'select',
    );
    $form += $this->fieldCase($settings);
    return $form;
  }

  /**
   * Validates form values.
   */
  function validate($form_values) {
    if ($form_values['term_select'] == 'select') {
      form_set_error('term_select', t('You must select a term.'));
    }
  }

  /**
   * Returns the link path.
   */
  function setPath($form_values) {
    $path = 'taxonomy/term/'. $form_values['term_select'];
    return $path;
  }

  /**
   * Returns an array of taxonomy terms for use in a select list.
   */
  private static function getTerms() {
    $terms = array();
    $result = db_query('SELECT * FROM {term_data}');
    while ($term = db_fetch_object($result)) {
      $terms[$term->tid] = $term->name;
    }
    asort($terms);
    return $terms;
  }
}

/**
 * Defines the multiple taxonomy term form class.
 */
class AutolinkTermMultipleForm extends AutolinkDefaultLinkForm {

  /**
   * Defines the link settings form.
   */
  function form($settings) {
    if (!empty($settings['path'])) {
      // Get the term names from tids by parsing the path.
      $args = explode('/', $settings['path']);
      $tids = explode(',', $args[2]);

      foreach ($tids as $tid) {
        $term = taxonomy_get_term($tid);
        $terms[$tid] = $term->name;
      }

      asort($terms);
      $terms_default = implode(', ', $terms);
    }

    $form = $this->fieldKeyword($settings);
    $form['terms'] = array(
      '#type' => 'textfield',
      '#title' => t('Taxonomy term(s)'),
      '#required' => TRUE,
      '#default_value' => isset($terms_default) ? $terms_default : '',
      '#description' => t('Enter a comma separated list of taxonomy terms to which the anchor should link.'),
    );
    $form['operator'] = array(
      '#type' => 'radios',
      '#title' => t('Operation'),
      '#options' => array(
        'and' => 'and',
        'or' => 'or',
      ),
      '#default_value' => isset($settings['operator']) ? $settings['operator'] : '',
    );
    $form += $this->fieldCase($settings);
    return $form;
  }

  /**
   * Validates form values.
   */
  function validate($form_values) {
    $terms = drupal_explode_tags($form_values['terms']);

    // Ensure an operation is selected if multiple terms were entered.
    if (isset($terms[1]) && empty($form_values['operator'])) {
      form_set_error('operator', t('You must select an operation for multiple term definitions.'));
    }
  
    // Check to ensure terms exist.
    foreach ($terms as $term) {
      if (self::termExists($term) == FALSE) {
        form_set_error('terms', t('Cannot locate term data for <i>%term</i>. Please enter another term.', array('%term' => $term)));
      }
    }
    unset($term);
  }

  /**
   * Returns the link path.
   *
   * Form elements are processed to build the proper taxonomy term
   * path for multiple term links.
   */
  function setPath($form_values) {
    // Get term IDs for use in urls.
    $terms = drupal_explode_tags($link['terms']);
    foreach ($terms as $name) {
      $result = db_query("SELECT * FROM {term_data} WHERE name = '%s'", $name);
      while ($term = db_fetch_object($result)) {
        $tids[] = $term->tid;
      }
    }

    // If this is only one term then we don't need an operation.
    if (!isset($terms[1])) {
      $arg = $tids[0];
    }
    else {
      // This is a multiple term url, so it needs to be built with operators (,) and (+).
      switch ($form_values['operator']) {
        case 'and':
          $arg = implode(',', $tids);
          break;
        case 'or':
          $arg = implode('+', $tids);
          break;
      }
    }

    $path = 'taxonomy/term/'. $arg;
    return $path;
  }

  /**
   * Checks if a taxonomy term exists.
   */
  private static function termExists($term) {
    $result = db_query("SELECT * FROM {term_data} WHERE name = '%s'", $term);
    return (db_fetch_object($result)) ? TRUE : FALSE;
  }
}

/**
 * Defines the node search form class.
 */
class AutolinkNodeSearchForm extends AutolinkDefaultLinkForm {

  /**
   * Defines the link settings form.
   */
  function form($settings) {
    $form = $this->fieldKeyword($settings);
    $form['search_terms'] = array(
      '#type' => 'textfield',
      '#title' => t('Search terms'),
      '#required' => TRUE,
      '#default_value' => isset($settings['query']) ? $settings['query'] : '',
      '#description' => t('The search terms whose results users should be directed upon clicking the link.'),
    );
    $form += $this->fieldCase($settings);
    return $form;
  }

  /**
   * Returns the link path.
   */
  function setPath($form_values) {
    return 'search/node/';
  }

  /**
   * Returns the path query.
   */
  function setQuery($form_values) {
    return $form_values['search_terms'];
  }

}

/**
 * Defines the user search form class.
 */
class AutolinkUserSearchForm extends AutolinkDefaultLinkForm {

  /**
   * Defines the link settings form.
   */
  function form($settings) {
    $form = $this->fieldKeyword($settings);
    $form['search_terms'] = array(
      '#type' => 'textfield',
      '#title' => t('Search terms'),
      '#required' => TRUE,
      '#default_value' => isset($settings['query']) ? $settings['query'] : '',
      '#description' => t('The search terms whose results users should be directed upon clicking the link.'),
    );
    $form += $this->fieldCase($settings);
    return $form;
  }

  /**
   * Returns the link path.
   */
  function setPath($form_values) {
    return 'search/user';
  }

  /**
   * Returns the URL query.
   */
  function setQuery($form_values) {
    return $form_values['search_terms'];
  }

}
