<?php

/**
 * Autolink provides hooks for defining plugins and link types to the
 * Autolink API. These plugins and link types are built by extending
 * base classes. This means that they require a knowledge of object-
 * oriented PHP programming. Following are definitions and explanaitions
 * of how you can interact with and customize the Autolink module with
 * your own code.
 */

/**
 * Defines a plugin to the Autolink API.
 *
 * @return
 *   An associative array of plugin information. The key of the array
 *   should be the machine-name of the plugin, with only lower-case
 *   letters and underscores. Available keys follow:
 *   -label: A translated label for use in administrative pages.
 *   -description: A translated string describing the primary linking
 *    functionality of the plugin. This string is used in administrative
 *    and filter descriptions.
 *   -class: The class that defines plugin. This class should extend
 *    the AutolinkPlugin class.
 *   -file: The file in which the plugin class can be found. This file
 *    should be indicated from the base directory of the module folder.
 *    If no file is specified, Autolink will check for modulename.autolink.inc.
 *    Autolink files are lazy loaded so defining classes in your .module
 *    file may cause not found errors.
 *   -method: Identifies the method by which processing is done.
 *    -filter: Uses hook_autolink_filter() to do processing.
 *    -node: Uses hook_autolink_node_{operation}() to do processing.
 *    -undefined: Uses a unique method to call autolink_process().
 *   -filter tips: An array of values indicating filter tip information.
 *    If a callback is used then the other values will be ignored.
 *    -callback: A callback function for retrieving the filter tip
 *     information. This function will be passed the $long value.
 *    -short: A single translated string representing the short tip.
 *    -long: A single translated string representing the long tip.
 *   -link types: An array of values indicating which link types are
 *    supported by the plugin.
 *   -config: An array of key => value pairs indicating how to handle
 *    the settings form. Modules may alternatively offer a settings form
 *    in a different area if desired.
 *    -class: The class that contains the configuration form. This class
 *     should extends the AutolinkConfigForm class and may implement
 *     form(), validate(), and submit() methods.
 *    -file: The file which holds the configuration class. This file
 *     should be relative to the base path of the defining module.
 *    -module: An optional module to use as the base path when loading
 *     include files.
 */
function hook_autolink_plugin_info() {
  return array(
    'autolink' => array(
      'label' => t('Default Autolink Plugin'),
      'description' => t('Converts pre-defined keywords to links in content.'),
      'class' => 'AutolinkDefaultPlugin',
      'file' => 'autolink_default.plugin.inc',
      'method' => 'filter',
      'filter tips' => array('callback' => 'autolink_filter_tip'),
      'link types' => array('external', 'node', 'user', 'taxonomy', 'node_search', 'user_search'),
      'config' => array('class' => 'AutolinkConfigFormDefault', 'file' => 'includes/autolink_default.admin.inc'),
    ),
  );
}

/**
 * Alters plugin data.
 *
 * The $plugins argument is an associative array of all of the plugin
 * data that was collected. This argument is passed by reference.
 */
function hook_autolink_plugin_info_alter(&$plugins) {
  // Remove the default Autolink plugin.
  unset($plugins['autolink']);
}

/**
 * Returns an array of fields to process.
 *
 * @param $type
 *   The node type of the node being processed.
 *
 * @return
 *   An array of values indicating which fields to process. Fields that
 *   are not named 'body' or 'teaser' will be assumed to be CCK fields.
 */
function autolink_field_callback($type) {
  return array('body', 'teaser', 'content_field');
}

/**
 * Defines an Autolink plugin class.
 *
 * Note: Link properties for the current link being processed may be
 * access by using the $this->get('property') method where the property
 * is data stored in the link type data.
 *
 * Properties:
 *   $_info - An array of plugin information as specified in Autolink hooks.
 *   $item - Contains the object being processed. This is normally a node.
 *   $config - The configuration object of the currently enabled plugin.
 *   $link_limit - A string or integer representing the link limit.
 *   $total_limit - A string or integer representing the total link limit.
 *
 * Methods:
 *   The following methods are used by the Autolink processor class during node
 *   processing. Plugins must define setter methods for setting attributes and
 *   interacting with the processor object.
 *   getLinkTypes();
 *   getTotalLimit();
 *   getLinkLimit();
 *   getExpression();
 *   buildReplacement();
 *
 * Methods that may be added and used by the processor class. Many of these
 * methods are not necessarily required, and some default methods exist.
 * Requirements depend on hook_autolink_plugin_info() settings, so see above.
 *   setExpression(AutolinkLink $link);
 *   setText(AutolinkLink $link);
 *   setTotalLimit($node);
 *   setLinkLimit();
 *   setCase(AutolinkLink $link);
 *   setAttributes(AutolinkLink $link);
 */
class SamplePlugin extends AutolinkPlugin {

  /**
   * Returns a regular expression used to search for keywords in text.
   *
   * @return
   *   An array of regular expression characters to prepend and append to the
   *   Autolink regular expression, with keys 'prepend' and 'append' respectively.
   */
  function getExpression(AutolinkLink $link) {
    return array('prepend' => '\b', 'append' => '\b');
  }

  /**
   * Returns the text of a link.
   *
   * If no setText() method is defined, the matched text will be used.
   * Link properties may be accessed by using the $link->get('property')
   * method.
   *
   * @param $match
   *   The text that matched the regular expression.
   *
   * @return
   *   A string to be used in the text of the anchor tag being built.
   *   To use the text that was found, simply return $match.
   */
  function getText(AutolinkLink $link, $match) {
    return $match;
  }

  /**
   * Returns a total limit for a given node.
   *
   * This method is defined and called by the AutolinkNodeProcessor
   * class and not the base processor.
   *
   * @param $node
   *   The node object of the node being processed.
   *
   * @return
   *   An integer representing the total number of links that may be
   *   generated across all enabled fields of the node, or 'all' to
   *   indicate that there is no total limit.
   */
  function getTotalLimit() {
    return variable_get('autolink_total_limit', 'all');
  }

  /**
   * Set the link limit per individual link.
   *
   * This method is defined and called by the AutolinkNodeProcessor
   * class and not the base processor.
   * If no method is implemented, the default setting of 'all' will be
   * used.
   */
  function getLinkLimit() {
    return variable_get('autolink_link_limit', 'all');
  }

  /**
   * Indicates whether the search for an individual link should be case sensitive.
   *
   * @return
   *   A boolean value whether to search for the keyword using case-sensitivity.
   *   TRUE indicates that a case-sensitive search should be used.
   */
  function getCase(AutolinkLink $link) {
    return $link->get('case') == 1;
  }

  /**
   * Returns an array of attributes to apply to a link.
   * If this method is not implemented, default Autolink attributes will be
   * applied.
   *
   * @return
   *   An associative array of link attributes to apply to the link
   *   being built. These attributes directly correspond to the attributes
   *   made available in the l() function. See API documentation for more
   *   information on Drupal functions.
   */
  function getAttributes(AutolinkLink $link) {
    return array(
      'title' => $this->get('title'),
      'rel' => variable_get('autolink_rel', 'nofollow'),
    );
  }

  /**
   * Returns which fields should be processed.
   * Note: this method is called by AutolinkNodeProcessor and not the
   * text processor. Additional processors can handle it appropriately.
   */
  function getFields() {
    return array('body', 'teaser');
  }
}

/**
 * Defines an Autolink plugin settings form class.
 *
 * The settings class may provide additional form elements to the Autolink
 * settings form. To add a form, the class needs only to implement a form()
 * method, and Autolink will add the returned form to the normal settings form.
 * Validation and submit handling is also possible through validate() and
 * submit() methods respectively.
 *
 * This class uses the magic __call() method to call methods that are
 * within the attached configuration object. Also available are magic
 * __get() and __set() methods, so most configuration object properties
 * and methods are available in public scope.
 *
 * Available (and relevant) methods:
 *   get($setting);
 *   getSettings($setting, array $array, $default = 0);
 *     -This method returns all enabled values of an array setting by
 *      looping through the setting and returning only those that are enabled
 *      and match a value in the input array. Essentially, it returns a copy
 *      of the input array ($array) with the disabled (not '1') values removed.
 *   set($setting, $value);
 *   fetchArraySettingSetting($setting);
 *     -This method returns an array settings object which can be used to
 *      check and manipulate settings created by multiple-value form elements.
 *   fetchSetting($setting, $default = 0);
 *     -This method returns a single setting object.
 *   save()
 *
 * Additional methods to be added by plugin configurations:
 *   form(&$form_state, $settings);
 *   validate($form, &$form_state);
 *   submit($form, &$form_state);
 *     -Note that the submit function is run before the Autolink submit.
 */
class SamplePluginConfigForm extends AutolinkConfigForm {

  /**
   * Defines an extra plugin specific configuration form.
   */
  protected function form(&$form_state, $settings) {
    // Demonstrates how to retrieve settings from the AutolinkConfig class.
    // Configuration objects are separate from the form class and can thus
    // be instantiated separately for other purposes. The following methods
    // are configuration object methods and not from the config form class.

    // Retrieve a single value setting.
    $this->get('setting', 'default');

    // Retrieve a setting array object and check if a value is enabled.
    $this->fetchArraySetting('setting')->isEnabled('key');

    // Retrieve a single value setting and print it to the screen. Note
    // that this is less efficient than just using the first example.
    $this->fetchSetting('setting', 'default')->printData();

    return $form;
  }

  /**
   * Validates the Autolink settings form.
   */
  protected function validate($form, &$form_state) {

  }

  /**
   * By default, Autolink sets variables in a manner similar to that of
   * system_settings_form(). For form elements that return array values,
   * Autolink stores the values in 1 or 0 format.
   */
  protected function submit($form, &$form_state) {

  }

}

/**
 * Defines link types to the Autolink API.
 *
 * @return
 *   An associative array of link type information where the key of the
 *   array is the machine-readable name of the link type.
 *   -label: The human-readable label of the link type. This is used in
 *    administrative forms.
 *   -class: The class which implements this link type. If no class is
 *    defined, the AutolinkLink class will be used. Link classes must
 *    extend the AutolinkLink class. Note: If no link class is defined,
 *    the standard AutolinkLink class will be used.
 *   -file: The file which contains the class that implements the link
 *    type. This file name should be from the root directory of the
 *    module folder. If no file is specified, Autolink will search for
 *    module_name.autolink.inc.
 *   -module: A module may be specified if paths for the link type are
 *    not relative to the defining module's base path. If they are then
 *    there is no need to specify module name. This string should use the
 *    machine-name of a module.
 *   -identifiable: A boolean value defining whether the link type can
 *    be identified. If it is identifiable, it may be enabled or disabled
 *    via the Autolink settings form. Link types that are not identifiable
 *    will always be assumed to be enabled. Unidentifiable link types must
 *    also be undefinable. Autolink constants AUTOLINK_DISABLED and
 *    AUTOLINK_ENABLED may also be used.
 *   -definable: A boolean value determining whether the link type can be
 *    defined. Definable link types must define at least one form class
 *    as forms are required with definable link types. Link types that
 *    are not definable must implement the AutolinkAbstractLink interface
 *    and use the getLinks method to provide links to Autolink. Autolink
 *    constants AUTOLINK_DISABLED and AUTOLINK_ENABLED may also be used.
 *   -provider: If a link type is marked as not definable, it must implement
 *    a 'provider' class to provide links to Autolink. Provider classes
 *    should be entirely separate from the standard link type class and
 *    should implement the AutolinkProvider interface, requiring a
 *    provideLinks() method which returns an array of sub-arrays detailing
 *    keywords and paths for each link. Standard link type class methods
 *    may be used via the link type class.
 *   -config: An associative array containing sub-arrays that define how
 *    to handle link definitions forms. This allows module to provide
 *    several different forms for the same link type. If multiple forms
 *    are provided, Autolink will display a selector to allow the user
 *    to select which form to display. This is implemented in the default
 *    Autolink plugin module, so see that module for examples.
 *    -label: The human-readable text discribing the form type. This text
 *     is displayed in the select list for link types that define more
 *     than one form.
 *    -class: The class that defines the form. Link form classes must extend
 *     the AutolinkLinkForm class.
 *    -file: The file in which the class can be found. This filename should
 *     be defined from the base directory of the defining module. If no
 *     file is specified, Autolink will search for module_name.autolink.inc.
 *    -module: A module may be specified to use the base path of another
 *     module when lazy-loading include files. This can allow the forms to
 *     be stored in another module file.
 */
function hook_autolink_link_type_info() {
  return array(
    'node' => array(
      'label' => t('Link to node'),
      'class' => 'AutolinkNodeLink',
      'file' => 'autolink_plugin.link_types.inc',
      'identifiable' => AUTOLINK_ENABLED,
      'definable' => AUTOLINK_ENABLED,
      'config' => array(
        'title' => array('label' => t('Select content'), 'class' => 'AutolinkNodeTitleForm', 'file' => 'autolink_default.forms.inc',
        ),
        'path' => array('label' => t('Enter a path'), 'class' => 'AutolinkNodePathForm', 'file' => 'autolink_default.forms.inc',
        ),
      ),
    ),
  );
}

/**
 * Alter link type information.
 */
function hook_autolink_link_type_info_alter(&$info) {
  // Add a form to the node link type's 'select a title' form.
  $info['node']['config']['title'] = array(
    'label' => t('Select content'),
    'class' => 'AutolinkNodeTitleForm',
    'file' => 'autolink_default.forms.inc',
  );
}

/**
 * Defines an Autolink link type class.
 *
 * Extending Autolink's AutolinkLink class is only optional for link
 * types in the event that data needs to be preprocessed.
 * Note: Link type classes may use $this->get('property') to retrieve
 * link type properties like keyword, path, or any other properties store
 * by the link type in the process() method.
 */
class SampleLinkType extends AutolinkLink {

}

/**
 * Defines an abstract link type.
 *
 * Abstract link types are implicitly defined by link type not being set
 * as "definable". Essentially, if a link cannot be defined through forms it
 * must be abstract. Abstract link types must provide a 'provider' class in
 * link type info which invokes the provideLinks() method. provideLinks should
 * return an array of links from the link type's AutolinkLink derived class.
 */
class AbstractLinkProvider extends AutolinkProvider {

  /**
   * Provides links to Autolink at runtime.
   *
   * @param $node
   *   The current node being processed.
   *
   * @return
   *   An array of arrays specifying link data. Each sub-array must contain
   *   at least a keyword and path.
   */
  function provideLinks($node) {
    // Database query to get node titles.
    $data = array();
    $data[] = array('keyword' => 'apples', 'path' => 'node/1');
    $data[] = array('keyword' => 'bananas', 'path' => 'node/2');
    return $data;
  }

}

/**
 * Defines a link form class.
 *
 * The AutolinkConfig object is also available in this class in the
 * $config propery.
 */
class SampleLinkForm extends AutolinkLinkForm {
  /**
   * Defines a link type form.
   *
   * @param $settings
   *   An associative array of data stored in the database for the link
   *   being edited. This only applies to edit forms, and these values
   *   should be used to set default values on form elements. Developers
   *   should use ternary operators to prevent warnings.
   *
   * @return
   *   An associative array of form elements, similar to other forms
   *   done in Drupal. The array does not need to specify the form wrappers
   *   and fieldsets that exist on the link form. The returned form will
   *   automatically be placed properly in the link_wrapper array.
   */
  function form($settings) {
    return $form;
  }

  /**
   * Validate form values. This is done in the same way as other Drupal forms.
   *
   * @param $form_values
   *   An associative array of form values. Nearly identical to $form_state['values'].
   */
  function validate($form_values) {
    form_set_error('keyword', t('You must enter a keyword.'));
  }

  /**
   * Returns a keyword for which to search. This method is required of all form classes.
   *
   * @param $form_values
   *   The $form_state['values'] array from the link form.
   *
   * @return
   *   A single value to store in the database as the keyword.
   */
  function setKeyword($form_values) {
    return $form_values['keyword'];
  }

  /**
   * Returns a path build from form values. This method is required of all form classes.
   *
   * @param $form_values
   *   The $form_state['values'] array from the link form.
   *
   * @return
   *   A single value to store in the database as the link path.
   */
  function setPath($form_values) {
    return $form_values['path'];
  }

  /**
   * Returns the URL query.
   */
  function setQuery($form_values) {
    // Taken from the node_search link type.
    return $form_values['search_terms'];
  }

  /**
   * Returns extra data for saving.
   *
   * @param $form_values
   *   The $form_state['values'] array from the link form.
   *
   * @return
   *   An associative array of values to save in the database. Any values
   *   that should be saved aside from keyword and path may be returned
   *   here. Values that are saved here will be accessible in the
   *   $this->get('property') methods of link and plugin objects.
   */
  function setExtra($form_values) {
    return array(
      'case' => $form_values['case_sensitivity'],
      'title' => $form_values['title'],
    );
  }
}

/**
 * Defines a processor class.
 *
 * Data processors may be defined to Autolink to handle specific data
 * in a unique way. The processor is called via the autolink_process()
 * function usually during Autolink nodeapi or filter hooks. Autolink
 * comes with a node processor and text processor, so check these out
 * before you decide to make one on your own.
 *
 * key: The machine-name of the processor. This is used as the argument in
 * autolink_process() to call the correct processor.
 *   -module: The defining module of the processor. This is used to
 *    determine the base path from which to load the file.
 *   -file: The file in which the processor class resides.
 *   -class: The processor class.
 */
function hook_autolink_processor_info() {
  return array(
    'node' => array(
      'module' => 'autolink',
      'file' => 'includes/autolink.processor.inc',
      'class' => 'AutolinkNodeProcessor',
    ),
  );
}

/**
 * Alters existing processor data.
 */
function hook_autolink_processor_info_alter(&$processors) {
  $processors['node']['class'] = 'MyNodeProcessor';
}

/**
 * Defines a processor class.
 *
 * This sample is taken from the AutolinkTextProcessor class.
 * Note that AutolinkProcessor does use a __call() method, which calls
 * plugin methods, so calls to plugin methods can be accomplished by
 * making direct method calls like $this->getFields().
 *
 * Note that the AutolinkProcessor class is abstract and requires that
 * child classes implement the following methods:
 * public function save();
 * protected function process($links);
 * public function clear($limit = AUTOLINK_DEFAULT_LIMIT, $link = NULL);
 */
class AutolinkSampleProcessor extends AutolinkProcessor {
  public $text = NULL;
  public $node = NULL;

  /**
   * Loads a node into the processor object for processing.
   */
  function load(&$text, $node) {
    $this->text = &$text;
    $this->node = $node;
    $this->plugin->setItem($node);
    return $this;
  }

  /**
   * Returns a processed node for saving.
   */
  function save() {
    return $this->text;
  }

  /**
   * Adds links to a node.
   */
  protected function process($links) {
    $total = 0;
    $updated = FALSE;
    $previous = $this->text;
    $this->processText($this->text, $total, $links);
    // Indicate whether the node has been changed.
    if ($text != $previous) {
      $updated = TRUE;
    }
    return $updated;
  }

  /**
   * Clears text of all Autolink links.
   *
   * @return
   *   TRUE if the node has changed. FALSE if not.
   */
  function clear($limit = AUTOLINK_DEFAULT_LIMIT, $link = NULL) {
    $this->clearText($this->text, $limit, $link);
  }
}

/**
 * Demonstrates the use of the autolink_process() function.
 */

// Loads a processor object. The argument passed identifies the type of
// processor as defined in hook_autolink_processor_info().
$processor = autolink_process('node');

// Loads data into the processor. Arguments depend on the processor being used.
$processor->load($node);

// Updates links in the text. Links are gathered based on supported link
// types according to the enabled plugin information.
$processor->update();

// Returns or saves the processed data. In the case of the node processor,
// the node is saved and returned. However, all Autolink processors pass
// processed data by reference, so the save() method is only necessary
// for inserting data.
$node = $processor->save();

// Removes existing links from text or node.
$processor->clear();

/**
 * Following are Autolink processing hooks. These may be used to notify
 * Autolink when to process nodes or text. They do not necessarily need to
 * be used, though, as autolink_process() can be called from anywhere. If
 * these hooks are used, only hooks implemented by the module that defines
 * the currently enabled plugin will be used. In other words, if these
 * hooks are implemented by disabled plugin modules the data will be ignored.
 */

/**
 * Modifies a node on presave.
 */
function hook_autolink_node_presave(&$node) {
  // Places links on a node.
  autolink_process('node')->load($node)->process();
}

/**
 * Modifies a node on insert.
 */
function hook_autolink_node_insert(&$node) {
  // Places links on a node.
  autolink_process('node')->load($node)->process();
}

/**
 * Modifies a node on update.
 */
function hook_autolink_node_update(&$node) {
  // Places links on a node.
  autolink_process('node')->load($node)->process();
}

/**
 * Modifies a node on prepare.
 */
function hook_autolink_node_prepare(&$node) {
  // Clears links from a node.
  autolink_process('node')->load($node)->clear();
}

/**
 * Modifies text during filtering.
 */
function hook_autolink_filter(&$text, $node) {
  autolink_process('text')->load($text, $node)->process();
}
