<?php

/**
 * @file
 * Administrative forms for autolink_ module.
 */

/**
 * Settings form for autolink.
 *
 * @ingroup forms
 * @see autolink_form_settings_submit()
 */
function autolink_form_settings(&$form_state, $edit = array()) {
  return autolink_get('config')->setForm()->execute('form', $form_state, $edit);
}

/**
 * Validation function for settings form.
 */
function autolink_form_settings_validate($form, &$form_state) {
  return autolink_get('config')->setForm()->execute('validate', $form, $form_state);
}

/**
 * Submit handler for Autolink settings form.
 *
 * This handler sets miltiple value field values by creating individual variables
 * for each value. This makes node and link type settings much easier to work with
 * on other forms and in processing functions and is cleaner than using the
 * 'array_filter' field.
 */
function autolink_form_settings_submit($form, &$form_state) {
  autolink_get('config')->setForm()->execute('submit', $form, $form_state);
}

/**
 * Callback for displaying the groups overview form.
 */
function autolink_admin_groups() {
  Autolink::loadFiles(array('filters', 'operations'));
  if (!empty($_POST['groups']) && isset($_POST['operation']) && ($_POST['operation'] == 'delete')) {
    $output = drupal_get_form('autolink_group_multiple_delete_confirm');
  }
  else {
    $output = drupal_get_form('autolink_filter_form');
    $output .= drupal_get_form('autolink_overview_groups');
  }
  return $output;
}

/**
 * Form builder for autolink_ groups overview filters.
 *
 * @ingroup forms
 * @see autolink_filter_form_submit()
 */
function autolink_filter_form() {
  $session = &$_SESSION['autolink_overview_filter'];
  $session = is_array($session) ? $session : array();
  $filters = autolink_filters();

  $i = 0;
  $form['filters'] = array(
    '#type' => 'fieldset',
    '#title' => t('Show only link groups where'),
    '#theme' => 'autolink_filters',
  );
  foreach ($session as $filter) {
    list($type, $value) = $filter;
    // Merge an array of arrays into one if necessary.
    $options = $filters[$type]['options'];
    $params = array('%property' => $filters[$type]['title'] , '%value' => $options[$value]);
    if ($i++ > 0) {
      $form['filters']['current'][] = array('#value' => t('<em>and</em> where <strong>%property</strong> is <strong>%value</strong>', $params));
    }
    else {
      $form['filters']['current'][] = array('#value' => t('<strong>%property</strong> is <strong>%value</strong>', $params));
    }
  }

  foreach ($filters as $key => $filter) {
    $names[$key] = $filter['title'];
    $form['filters']['status'][$key] = array(
      '#type' => 'select',
      '#options' => $filter['options'],
    );
  }

  $form['filters']['filter'] = array(
    '#type' => 'radios',
    '#options' => $names,
  );
  $form['filters']['buttons']['submit'] = array(
    '#type' => 'submit',
    '#value' => (count($session) ? t('Refine') : t('Filter')),
  );

  if (count($session)) {
    $form['filters']['buttons']['undo'] = array(
      '#type' => 'submit',
      '#value' => t('Undo'),
    );
    $form['filters']['buttons']['reset'] = array(
      '#type' => 'submit',
      '#value' => t('Reset'),
    );
  }

  drupal_add_js('misc/form.js', 'core');

  return $form;
}

/**
 * Theme autolink_ administration filter form.
 *
 * @ingroup themeable
 */
function theme_autolink_filter_form($form) {
  $output = '<div id="autolink-admin-filter">';
  $output .= drupal_render($form['filters']);
  $output .= '</div>';
  $output .= drupal_render($form);
  return $output;

}

/**
 * Theme autolink administration filter selector.
 *
 * @ingroup themeable
 */
function theme_autolink_filters($form) {
  $output = '<ul class="clear-block">';

  if (!empty($form['current'])) {
    foreach (element_children($form['current']) as $key) {
      $output .= '<li>'. drupal_render($form['current'][$key]) .'</li>';
    }
  }

  $output .= '<li><dl class="multiselect">'. (!empty($form['current']) ? '<dt><em>'. t('and') .'</em> '. t('where') .'</dt>' : '') .'<dd class="a">';

  foreach (element_children($form['filter']) as $key) {
    $output .= drupal_render($form['filter'][$key]);
  }

  $output .= '</dd>';
  $output .= '<dt>'. t('is') .'</dt><dd class="b">';

  foreach (element_children($form['status']) as $key) {
    $output .= drupal_render($form['status'][$key]);
  }

  $output .= '</dd>';
  $output .= '</dl>';

  $output .= '<div class="container-inline" id="autolink-admin-buttons">'. drupal_render($form['buttons']) .'</div>';
  $output .= '</li></ul>';

  return $output;
}


/**
 * Process result from autolink administration filter form.
 */
function autolink_filter_form_submit($form, &$form_state) {
  $op = $form_state['values']['op'];
  $filters = autolink_filters();

  switch ($op) {
    case t('Filter'): case t('Refine'):
      if (isset($form_state['values']['filter'])) {
        $filter = $form_state['values']['filter'];
        $options = $filters[$filter]['options'];
        if (isset($options[$form_state['values'][$filter]])) {
          $_SESSION['autolink_overview_filter'][] = array($filter, $form_state['values'][$filter]);
        }
      }
      break;
    case t('Undo'):
      array_pop($_SESSION['autolink_overview_filter']);
      break;
    case t('Reset'):
      $_SESSION['autolink_overview_filter'] = array();
      break;
    case t('Update'):
      return;
  }

  $form_state['redirect'] = 'admin/content/autolink/';
  return;
}

/**
 * Form builder; autolink_ administration page.
 *
 * @ingroup forms
 * @see autolink_overview_groups_validate()
 * @see autolink_overview_groups_submit()
 */
function autolink_overview_groups() {
  $filter = autolink_build_filter_query();

  $header = array();
  $header[] = array('data' => t('Group'), 'field' => 'g.name', 'sort' => 'asc');
  $header[] = array('data' => t('Description'), 'field' => 'g.description');
  $header[] = t('Node Types');
  $header[] = array('data' => t('Status'), 'field' => 'g.status');
  $header[] = t('Operations');

  if ($filter['join'] != "") {
    $sql = 'SELECT DISTINCT g.type, g.name, g.description, g.status FROM {autolink_group} g LEFT JOIN {autolink_group_node_type} t ON g.type = t.group_type '. $filter['join'] .' '. $filter['where'];
    $query_count = 'SELECT COUNT(DISTINCT g.type) FROM {autolink_group} g LEFT JOIN {autolink_group_node_type} t ON g.type = t.group_type '. $filter['join'] .' '. $filter['where'];
  }
  else {
    $sql = 'SELECT g.type, g.name, g.description, g.status FROM {autolink_group} g '. $filter['where'];
    $query_count = 'SELECT COUNT(g.type) FROM {autolink_group} g '. $filter['where'];
  }

  $sql .= tablesort_sql($header);

  $result = pager_query($sql, 25, 0, $query_count, $filter['args']);

  $form['options'] = array(
    '#type' => 'fieldset',
    '#title' => t('Update link groups'),
    '#prefix' => '<div class="container-inline">',
    '#suffix' => '</div>',
  );

  // Add our own operations first.
  $options = array();
  $ops = autolink_group_operations();
  foreach ($ops as $operation => $array) {
    $options[$operation] = $array['label'];
  }

  // Invoke hook_autolink_operations() to get a list of external operations.
  foreach (module_invoke_all('autolink_group_operations') as $operation => $array) {
    $options[$operation] = $array['label'];
  }

  $form['options']['operation'] = array(
    '#type' => 'select',
    '#options' => $options,
    '#default_value' => 'activate',
  );
  $form['options']['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Update'),
  );

  $destination = drupal_get_destination();

  $status = array(t('disabled'), t('active'));
  $groups = array();

  while ($group = db_fetch_object($result)) {
    $groups[$group->type] = '';
    $form['group'][$group->type] = array('#value' => l($group->name, "admin/content/autolink/$group->type"));
    $form['description'][$group->type] = array('#value' => $group->description);

    $node_types = array();
    $types_result = db_query("SELECT n.type, n.name FROM {node_type} n INNER JOIN {autolink_group_node_type} t ON n.type = t.node_type WHERE t.group_type = '%s'", $group->type);
    while ($type = db_fetch_object($types_result)) {
      $node_types[$type->type] = $type->name;
    }
    asort($node_types);
    $form['types'][$group->type][0] = array('#value' => implode(', ', $node_types));
    $form['status'][$group->type] =  array('#value' => $status[$group->status]);
    $form['operations'][$group->type]['edit'] = array('#value' => l(t('edit group'), "admin/content/autolink/edit/group/$group->type", array('query' => $destination)));
    $form['operations'][$group->type]['add'] = array('#value' => l(t('add link'), "admin/content/autolink/$group->type/add/link", array('query' => "admin/content/autolink/$group->type")));
  }

  $form['groups'] = array(
    '#type' => 'checkboxes',
    '#options' => $groups
  );
  $form['pager'] = array('#value' => theme('pager', NULL, 25, 0));

  return $form;
}

/**
 * Form validation for autolink_ groups administration update form.
 */
function autolink_overview_groups_validate($form, &$form_state) {
  $form_state['values']['groups'] = array_filter($form_state['values']['groups']);
  if (!count($form_state['values']['groups'])) {
    form_set_error('', t('No groups were selected.'));
  }
}

/**
 * Submit handler for autolink_ groups administration update form.
 */
function autolink_overview_groups_submit($form, &$form_state) {
  $operations = autolink_group_operations($form_state);
  $operations += module_invoke_all('autolink_group_operations', $form_state);
  $operation = $operations[$form_state['values']['operation']];
  // Filter out unchecked groups.
  $groups = array_filter($form_state['values']['groups']);
  if ($function = $operation['callback']) {
    // Add in callback arguments if present.
    if (isset($operation['callback arguments'])) {
      $args = array_merge(array($groups), $operation['callback arguments']);
    }
    else {
      $args = array($groups);
    }
    // Call the function based on the operation that was selected.
    call_user_func_array($function, $args);

    drupal_set_message(t('The update has been performed.'));
  }
}

/**
 * Theme autolink_ administration groups overview.
 *
 * @ingroup themeable
 */
function theme_autolink_overview_groups($form) {
  // Overview table:
  $header = array(
    theme('table_select_header_cell'),
    array('data' => t('Group'), 'field' => 'g.type', 'sort' => 'asc'),
    array('data' => t('Description'), 'field' => 'g.description'),
    t('Content types'),
    array('data' => t('Status'), 'field' => 'g.status'),
    t('Operations'),
  );

  $output = drupal_render($form['options']);
  if (isset($form['group']) && is_array($form['group'])) {
    foreach (element_children($form['group']) as $key) {
      $rows[] = array(
        drupal_render($form['groups'][$key]),
        drupal_render($form['group'][$key]),
        drupal_render($form['description'][$key]),
        drupal_render($form['types'][$key]),
        drupal_render($form['status'][$key]),
        drupal_render($form['operations'][$key]),
      );
    }
  }
  else {
    $rows[] = array(array('data' => t('No link groups available.'), 'colspan' => '6'));
  }

  $output .= theme('table', $header, $rows);
  if ($form['pager']['#value']) {
    $output .= drupal_render($form['pager']);
  }

  $output .= drupal_render($form);

  return $output;
}

/**
 * Display form for adding and editing link groups.
 *
 * @ingroup forms
 * @see autolink_form_group_submit()
 */
function autolink_form_group(&$form_state, $edit = array()) {
  $form['identification'] = array(
    '#type' => 'fieldset',
    '#title' => t('Identification'),
    '#collapsible' => TRUE,
  );
  $form['identification']['status'] = array(
    '#type' => 'select',
    '#title' => t('Status'),
    '#default_value' => isset($edit['status']) ? $edit['status'] : 1,
    '#options' => array(0 => t('Disabled'), 1 => t('Active')),
  );
  $form['identification']['name'] = array(
    '#type' => 'textfield',
    '#title' => t('Group name'),
    '#default_value' => isset($edit['name']) ? $edit['name'] : '',
    '#maxlength' => 255,
    '#description' => t('The human-readable name for this link group.'),
    '#required' => TRUE,
  );
  $form['identification']['description'] = array(
    '#type' => 'textarea',
    '#title' => t('Description'),
    '#default_value' => isset($edit['description']) ? $edit['description'] : '',
    '#description' => t('Description of the link group.'),
  );
  $form['identification']['in_content'] = array(
    '#type' => 'checkbox',
    '#title' => t('Generate links from this group in node content'),
    '#default_value' => isset($edit['in_content']) ? $edit['in_content'] : '',
    '#description' => t('autolink_ will search for keywords from this link group in '.
                        'node content and replace them with proper links.'),
  );
  $form['nodes'] = array(
    '#type' => 'fieldset',
    '#title' => t('Content types'),
    '#collapsible' => TRUE,
  );
  $form['nodes']['node_types'] = array(
    '#type' => 'checkboxes',
    '#title' => t('Content types'),
    '#default_value' => isset($edit['types']) ? $edit['types'] : array(),
    '#options' => array_map('check_plain', node_get_types('names')),
    '#description' => t('Select content types for this link group to monitor.'),
  );

  $form['submit'] = array('#type' => 'submit', '#value' => t('Save'));
  if (isset($edit['type'])) {
    $form['delete'] = array('#type' => 'submit', '#value' => t('Delete'));
    $form['type'] = array('#type' => 'value', '#value' => $edit['type']);
  }
  else {
    // Add a validate function to ensure group names are not duplicated.
    $form['#valdate'][] = 'autolink_form_group_duplicate_validate';
  }

  return $form;
}

/**
 * Validate function for add link group form.
 */
function autolink_form_group_duplicate_validate($form, &$form_state) {
  $group = $form_state['values']['name'];
  $result = db_query("SELECT * FROM {autolink_group} WHERE name = '%s'", $group);
  if (db_result($result)) {
    form_set_error('name', t('A link group with the name %name already exists.', array('%name' => $group)));
  }
}

/**
 * Submit function for add link group form.
 */
function autolink_form_group_submit($form, &$form_state) {
  // Fix up the nodes array to remove unchecked nodes.
  $form_state['values']['types'] = array_filter($form_state['values']['node_types']);
  switch (autolink_group_save($form_state['values'])) {
    case SAVED_NEW:
      drupal_set_message(t('Created new link group %name.', array('%name' => $form_state['values']['name'])));
      watchdog('autolink', 'Created new link group %name.', array('%name' => $form_state['values']['name']), WATCHDOG_NOTICE, l(t('edit'), 'admin/content/autolink/edit/group/'. $form_state['values']['type']));
      break;
    case SAVED_UPDATED:
      drupal_set_message(t('Updated link group %name.', array('%name' => $form_state['values']['name'])));
      watchdog('autolink', 'Updated link group %name.', array('%name' => $form_state['values']['name']), WATCHDOG_NOTICE, l(t('edit'), 'admin/content/autolink/edit/group/'. $form_state['values']['type']));
      break;
  }

  $form_state['type'] = $form_state['values']['type'];
  $form_state['redirect'] = 'admin/content/autolink';
  return;
}

/**
 * Page to edit a link group.
 */
function autolink_admin_group_edit($group) {
  if ((isset($_POST['op']) && $_POST['op'] == t('Delete')) || isset($_POST['confirm'])) {
    return drupal_get_form('autolink_group_confirm_delete', $group->type);
  }
  return drupal_get_form('autolink_form_group', (array)$group);
}

/**
 * Form builder for the link group delete confirmation form.
 *
 * @ingroup forms
 * @see autolink_group_confirm_delete_submit()
 */
function autolink_group_confirm_delete(&$form_state, $type) {
  $group = autolink_group_load($type);

  $form['type'] = array('#type' => 'value', '#value' => $type);
  $form['name'] = array('#type' => 'value', '#value' => $group->name);
  return confirm_form($form,
                  t('Are you sure you want to delete the link group %name?',
                  array('%name' => $group->name)),
                  'admin/content/autolink',
                  t('Deleting a link group will delete all the link definitions within it. This action cannot be undone.'),
                  t('Delete'),
                  t('Cancel'));
}

/**
 * Submit handler to delete a link group after confirmation.
 *
 * @see autolink_group_confirm_delete()
 */
function autolink_group_confirm_delete_submit($form, &$form_state) {
  $status = autolink_group_delete($form_state['values']['type']);
  drupal_set_message(t('Deleted link group %name.', array('%name' => $form_state['values']['name'])));
  watchdog('autolink', 'Deleted link group %name.', array('%name' => $form_state['values']['name']), WATCHDOG_NOTICE);
  $form_state['redirect'] = 'admin/content/autolink';
  return;
}

/**
 * Callback for generating the links overview form.
 */
function autolink_admin_links($list, $group) {
  Autolink::loadFiles(array('filters', 'operations'));
  if (!empty($_POST['links']) && isset($_POST['operation']) && ($_POST['operation'] == 'delete')) {
    $output = drupal_get_form('autolink_link_multiple_delete_confirm');
  }
  else {
    drupal_set_title(t('Link definitions for %group', array('%group' => $group->name)));
    $output = drupal_get_form('autolink_overview_links', (array) $group);
  }
  return $output;
}

/**
 * Form builder; autolink_ administration page.
 *
 * @ingroup forms
 * @see autolink_overview_links_validate()
 * @see autolink_overview_links_submit()
 */
function autolink_overview_links($form_state, $group) {
  Autolink::loadFiles(array('config'));
  $header = array();
  $header[] = array('data' => t('Type'), 'field' => 'a.type');
  $header[] = array('data' => t('Keyword'), 'field' => 'a.keyword', 'sort' => 'asc');
  $header[] = t('Destination');
  $header[] = t('Operations');

  $sql = "SELECT a.lid, a.group_type, a.type, a.keyword, a.path, a.query FROM {autolink_link} a WHERE a.group_type = '". $group['type'] ."' AND a.lid <> 0";
  $query_count = "SELECT COUNT(a.lid) FROM {autolink_link} a WHERE a.group_type = '". $group['type'] ."' AND a.lid <> 0";

  $sql .= tablesort_sql($header);

  $result = pager_query($sql, 25, 0, $query_count, $filter['args']);

  $form['options'] = array(
    '#type' => 'fieldset',
    '#title' => t('Update options'),
    '#prefix' => '<div class="container-inline">',
    '#suffix' => '</div>',
  );
  $options = array();

  // Add our own operations first.
  $ops = autolink_link_operations();
  foreach ($ops as $operation => $array) {
    $options[$operation] = $array['label'];
  }

  // Invoke hook_autolink_operations() to get a list of external operations.
  foreach (module_invoke_all('autolink_link_operations') as $operation => $array) {
    $options[$operation] = $array['label'];
  }

  $form['options']['operation'] = array(
    '#type' => 'select',
    '#options' => $options,
    '#default_value' => 'activate',
  );
  $form['options']['submit'] = array(
    '#type' => 'submit',
    '#value' => t('Update'),
  );

  $destination = drupal_get_destination();

  $links = array();
  $link_types = autolink_get('LinkTypeInfo')->getDefinable('label');
  while ($link = db_fetch_object($result)) {
    $destination = drupal_lookup_path('alias', $link->path) ? drupal_lookup_path('alias', $link->path) : $link->path;
    if ($link->type == 'node_search' || $link->type == 'user_search') {
      $destination = $link->path .'/'. $link->query;
    }

    // Link the keyword to its relationship page if tracking is enabled.
    $keyword = module_exists('autolink_tracking') ? l($link->keyword, "admin/content/autolink/$link->group_type/link/$link->lid") : $link->keyword;

    $links[$link->lid] = '';
    $form['type'][$link->lid] = array('#value' => $link_types[$link->type]);
    $form['keyword'][$link->lid] = array('#value' => $keyword);
    $form['destination'][$link->lid] =  array('#value' => l($destination, $link->path));
    $form['operations'][$link->lid] = array('#value' => l(t('edit'), "admin/content/autolink/$link->group_type/edit/link/$link->lid", array('query' => "admin/content/autolink/$link->group_type")));
  }

  $form['links'] = array(
    '#type' => 'checkboxes',
    '#options' => $links
  );
  $form['pager'] = array('#value' => theme('pager', NULL, 25, 0));
  $form['group'] = array('#type' => 'value', '#value' => $group['name']);

  return $form;
}

/**
 * Form validation for autolink_ administration update form.
 */
function autolink_overview_links_validate($form, &$form_state) {
  $form_state['values']['links'] = array_filter($form_state['values']['links']);
  if (!count($form_state['values']['links'])) {
    form_set_error('', t('No link definitions were selected.'));
  }
}

/**
 * Submit handler for autolink_ administration update form.
 */
function autolink_overview_links_submit($form, &$form_state) {
  $operations = autolink_link_operations($form_state);
  $operations += module_invoke_all('autolink_link_operations', $form_state);
  $operation = $operations[$form_state['values']['operation']];
  // Filter out unchecked links.
  $links = array_filter($form_state['values']['links']);
  if ($function = $operation['callback']) {
    // Add in callback arguments if present.
    if (isset($operation['callback arguments'])) {
      $args = array_merge(array($links), $operation['callback arguments']);
    }
    else {
      $args = array($links);
    }
    // Call the function based on the operation that was selected.
    call_user_func_array($function, $args);

    drupal_set_message(t('The update has been performed.'));
  }
}

/**
 * Theme autolink_ administration overview.
 *
 * @ingroup themeable
 */
function theme_autolink_overview_links($form) {
  // Overview table:
  $header = array(
    theme('table_select_header_cell'),
    array('data' => t('Type'), 'field' => 'a.type'),
    array('data' => t('Keyword'), 'field' => 'a.keyword', 'sort' => 'asc'),
    t('Destination'),
    t('Operations'),
  );

  $output = drupal_render($form['options']);
  if (isset($form['keyword']) && is_array($form['keyword'])) {
    foreach (element_children($form['keyword']) as $key) {
      $rows[] = array(
        drupal_render($form['links'][$key]),
        drupal_render($form['type'][$key]),
        drupal_render($form['keyword'][$key]),
        drupal_render($form['destination'][$key]),
        drupal_render($form['operations'][$key]),
      );
    }
  }
  else {
    $rows[] = array(array('data' => t('No link definitions available.'), 'colspan' => '5'));
  }

  $output .= theme('table', $header, $rows);
  if ($form['pager']['#value']) {
    $output .= drupal_render($form['pager']);
  }

  $output .= drupal_render($form);

  return $output;
}

/**
 * Form for adding or editing a link definition.
 *
 * This function uses the AHAH Helper module to build a dynamic link definition
 * form. When the AHAH submit function is called and the form is rebuilt this
 * function builds link type specific form elements based on data retrieved with
 * the link type that has been selected.
 */
function autolink_form_link(&$form_state, $group, $settings = array()) {
  Autolink::loadFiles(array('config', 'forms'));
  $info = autolink_get('LinkTypeInfo');
  $config = autolink_get('Config');

  $form = array();
  $form['group'] = array('#type' => 'value', '#value' => $group->type);

  ahah_helper_register($form, $form_state);

  // Prepare link types for the form.
  $select_type = array('select' => t('--Select a link type--'));
  $link_types = $config->getSettings('link_types', $info->getDefinable('label'), 1);
  asort($link_types);

  if (!empty($link_types)) {
    $link_type_options = array_merge($select_type, $link_types);
  }
  else {
    $link_type_options = $select_type;
    drupal_set_message(t('There are currently no link types enabled. Link types may be enabled by visiting the setting page.'), 'warning');
  }

  $keyword_disabled = FALSE;
  $keyword_required = TRUE;

  if (!isset($form_state['storage']['link_wrapper']['type'])) {
    $link_type = isset($settings['type']) ? $settings['type'] : 'select';
  }
  else {
    $link_type = $form_state['storage']['link_wrapper']['type'];
  }

  $form['link_wrapper'] = array(
    '#prefix' => '<div id="link-wrapper">',
    '#suffix' => '</div>',
    '#tree'   => TRUE,
  );
  // If there is only one link type then set that as the default instead of 'select'.
  $form['link_wrapper']['type'] = array(
    '#type' => 'select',
    '#title' => t('Link to'),
    '#options' => $link_type_options,
    '#default_value' => $link_type,
    '#ahah' => array(
      'event' => 'change',
      'path' => ahah_helper_path(array('link_wrapper')),
      'wrapper' => 'link-wrapper',
      'effect' => 'fade',
    ),
  );
  $form['link_wrapper']['type_submit'] = array(
    '#type' => 'submit',
    '#value' => t('Set link type'),
    '#submit' => array('ahah_helper_generic_submit'),
    '#attributes' => array('class' => 'no-js'),
  );

  // Only if a link type has been selected do we show the link definition form.
  if (!empty($link_type) && $link_type != 'select') {
    $form['link_wrapper']['definition'] = array(
      '#type' => 'fieldset',
      '#title' => t('Definition'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    );
    $link_type_info = $info->getLinkTypeInfo($link_type, 'config');

    // Display an additional selector if there are multiple forms for the link type.
    if (count($link_type_info) > 1) {
      // Set the method for edit forms or from the form storage.
      if (!isset($form_state['storage']['link_wrapper']['definition']['method_wrapper']['method'])) {
        $selected_method = isset($settings['method']) ? $settings['method'] : 'no_method';
      }
      else {
        $selected_method = $form_state['storage']['link_wrapper']['definition']['method_wrapper']['method'];
      }

      $method_options['no_method'] = t('--Select a method--');
      $options = array();
      foreach ($info->getLinkTypeInfo($link_type, 'config') as $key => $value) {
        $options[$key] = $value['label'];
      }
      $method_options += $options;

      $form['link_wrapper']['definition']['method_wrapper'] = array(
        '#prefix' => '<div id="method-wrapper">',
        '#suffix' => '</div>',
        '#tree' => TRUE,
      );
      $form['link_wrapper']['definition']['method_wrapper']['method'] = array(
        '#type' => 'select',
        '#title' => t('Selection method'),
        '#options' => $method_options,
        '#default_value' => $selected_method,
        '#required' => TRUE,
        '#ahah' => array(
          'event' => 'change',
          'path' => ahah_helper_path(array('link_wrapper', 'definition', 'method_wrapper')),
          'wrapper' => 'method-wrapper',
          'effect' => 'fade',
        ),
      );

      if ($selected_method != 'no_method') {
        $type_form = AutolinkFormFactory::getInstance($link_type, $selected_method);
        $form['link_wrapper']['definition']['method_wrapper'] += $type_form->execute('form', $settings);
      }
    }
    else {
      $selected_method = key($link_type_info);
      $form['method'] = array('#type' => 'value', '#value' => $selected_method);
      $type_form = AutolinkFormFactory::getInstance($link_type, $selected_method);
      $form['link_wrapper']['definition'] += $type_form->execute('form', $settings);
    }
  }

  $form['submit'] = array('#type' => 'submit', '#value' => t('Save'));
  $form['group_type'] = array('#type' => 'value', '#value' => $group->type);
  if (isset($settings['lid'])) {
    $form['delete'] = array('#type' => 'submit', '#value' => t('Delete'));
    $form['lid'] = array('#type' => 'value', '#value' => $settings['lid']);
  }
  return $form;
}

/**
 * Validation handler for the link edit form.
 *
 * Here we ensure the link does not contain inappropriate characters. Also, since
 * we are using disabled and hidden fields, we have to do extra form validation
 * here because we cannot use the 'required' => TRUE property on a disabled field.
 *
 * @see autolink_form_link()
 */
function autolink_form_link_validate($form, &$form_state) {
  Autolink::loadFiles(array('forms'));
  $values = _autolink_merge_link_form_values($form_state['values']);
  AutolinkFormFactory::getInstance($values['type'], $values['method'])->execute('validate', $values);
}

/**
 * Final submit function for the link form.
 */
function autolink_form_link_submit($form, &$form_state) {
  Autolink::loadFiles(array('forms'));
  $values = _autolink_merge_link_form_values($form_state['values']);
  $object = AutolinkFormFactory::getInstance($values['type'], $values['method']);

  switch ($object->execute('save', $values)) {
    case SAVED_NEW:
      drupal_set_message(t('Created new <i>%link_type</i> type link for keyword <i>%keyword</i>.', array('%link_type' => $values['type'], '%keyword' => $values['keyword'])));
      watchdog('autolink', 'Created new <i>%link_type</i> type link for keyword <i>%keyword</i>.', array('%link_type' => $values['type'], '%keyword' => $values['keyword']), WATCHDOG_NOTICE, l(t('edit'), 'admin/content/autolink/'. $values['group_type'] .'/edit/link/'. $link['lid']));
      break;
    case SAVED_UPDATED:
      drupal_set_message(t('Updated <i>%link_type</i> type link for keyword <i>%keyword</i>.', array('%link_type' => $values['type'], '%keyword' => $values['keyword'])));
      watchdog('autolink', 'Updated link for keyword <i>%keyword</i>.', array('%keyword' => $values['keyword']), WATCHDOG_NOTICE, l(t('edit'), 'admin/content/autolink/'. $values['group_type'] .'/edit/link/'. $values['lid']));
      break;
  }

  $form_state['lid'] = $form_state['values']['lid'];
  drupal_goto('admin/content/autolink/'. $values['group_type']);

  return;
}

/**
 * Page to add a link.
 */
function autolink_add_link_page($group) {
  drupal_set_title(t('Add link definition to group %group', array('%group' => $group->name)));
  return drupal_get_form('autolink_form_link', $group);
}

/**
 * Page to edit a link.
 *
 * @param $link
 *   A link object.
 *
 * @see autolink_form_link()
 * @see autolink_confirm_delete()
 */
function autolink_admin_link_edit($group, $lid) {
  if ((isset($_POST['op']) && $_POST['op'] == t('Delete')) || isset($_POST['confirm'])) {
    return drupal_get_form('autolink_confirm_link_delete', $lid);
  }
  Autolink::loadFiles(array('links'));
  $link = autolink_get('LinkMapper')->load($lid);
  return drupal_get_form('autolink_form_link', $group, (array)$link);
}

/**
 * Form builder for the link delete confirmation form.
 *
 * @param $lid
 *   The ID of a link.
 *
 * @ingroup forms
 * @see autolink_confirm_link_delete_submit()
 */
function autolink_confirm_link_delete(&$form_state, $lid) {
  $data = autolink_get('LinkMapper')->load($lid);

  $form['type'] = array(
    '#type' => 'value',
    '#value' => 'link',
  );
  $form['lid'] = array(
    '#type' => 'value',
    '#value' => $data['lid'],
  );
  $form['keyword'] = array(
    '#type' => 'value',
    '#value' => $data['keyword'],
  );
  return confirm_form($form,
                  t('Are you sure you want to delete the link %keyword?',
                  array('%keyword' => $data['keyword'])),
                  "admin/content/autolink/$link->group_type",
                  t('This action cannot be undone.'),
                  t('Delete'),
                  t('Cancel'));
}

/**
 * Submit handler to delete a link after confirmation.
 *
 * @see autolink_confirm_link_delete()
 */
function autolink_confirm_link_delete_submit($form, &$form_state) {
  // Delete the link and record the deletion
  $status = autolink_get('LinkMapper')->delete($form_state['values']['lid']);

  drupal_set_message(t('Deleted link definition for keyword <i>%link</i>.', array('%link' => $form_state['values']['keyword'])));
  watchdog('autolink', 'Deleted link definition for keyword <i>%link</i>.', array('%link' => $form_state['values']['keyword']), WATCHDOG_NOTICE);
  $form_state['redirect'] = 'admin/content/autolink';

  return;
}

/**
 * Merges AHAH form values into a readable $link array in preparation for database insert.
 *
 * @return
 *   Merged form values that can be processed for validation or inserted into the database.
 */
function _autolink_merge_link_form_values($form_values) {
  // Because of AHAH we need to add the proper form elements to a link array of form data.
  $link = $form_values;
  $link += $form_values['link_wrapper'];
  $link += $form_values['link_wrapper']['definition'];

  foreach ($form_values['link_wrapper']['definition'] as $key => $value) {
    if ($key == 'method_wrapper' || $value['#type'] == 'fieldset') {
      $link += $form_values['link_wrapper']['definition'][$key];
    }
  }

  return $link;
}
